import IAjaxCommand = drupal.Core.IAjaxCommand;
import IAjax = drupal.Core.IAjax;
import Location from "./map/Location";
import ProviderMapBase from "./map/providers/ProviderMapBase";
import MapPoint from "./map/MapPoint";
import MapIcon from "./map/MapIcon";
import Subscriber from "./map/Subscriber";
import {Marker} from "./map/Marker";
import MarkerList from "./map/MarkerList";

interface ProviderMapInterface {

  /**
   * The root element for the map.
   */
  element: HTMLElement

  /**
   * The map settings.
   */
  settings: ProviderMapSettings

  /**
   * The map instance for a specific map provider.
   */
  map: any

  /**
   * Sets the center of the map to the specified point.
   *
   * @param {drupalMap.MapPointInterface} point - The geographical point to set as the map's center.
   */
  setCenter(point: drupalMap.MapPointInterface): void

  /**
   * Retrieves the current zoom level.
   *
   * @return {number} The current zoom level as a numeric value.
   */
  getZoom(): number

  /**
   * Sets the zoom level for the view.
   *
   * @param {number} zoom - The desired zoom level. Must be a positive number.
   */
  setZoom(zoom: number): void

  /**
   * Adds an event listener for the specified map event.
   * The callback function will be invoked when the event is triggered.
   *
   * @param {string} event - The name of the map event to listen for.
   * @param {function} callback - The function to be executed when the event is triggered. It can accept variable arguments depending on the event type.
   */
  addMapEventListener(event: string, callback: (...args: any[]) => void): void

  /**
   * Adjusts and sets the boundaries of the map view based on the provided markers.
   *
   * @param {any[]} markers - An array of marker objects used to determine the boundaries.
   */
  setBoundsFromMarkers(markers: any[]): void

  /**
   * Sets the bounds of a map based on an array of map points.
   *
   * @param {drupalMap.MapPointInterface[]} points - An array of map points used to calculate the bounds.
   */
  setBoundsFromMapPoints(points: drupalMap.MapPointInterface[]): void

  /**
   * Adds a marker to the map with optional configuration.
   *
   * @param {drupalMap.MarkerInterface|drupalMap.MapPointInterface} marker - The marker object or map point to be added.
   * @param {boolean} [enabled] - Optional parameter indicating whether the marker should be enabled. Defaults to false if not provided.
   *
   * @return {any} Returns a reference to the added marker or the result of the operation.
   */
  addMarker(marker: drupalMap.MarkerInterface|drupalMap.MapPointInterface, enabled?: boolean): any

  /**
   * Enables the specified marker, making it active and visible on the map or its corresponding interface.
   *
   * @param {drupalMap.MarkerInterface} marker - The marker object to be enabled.
   */
  enableMarker(marker: drupalMap.MarkerInterface): void

  /**
   * Disables the specified marker, making it inactive or removing its
   * visibility/interaction on the map.
   *
   * @param {drupalMap.MarkerInterface} marker - The marker object that needs to
   *   be disabled.
   */
  disableMarker(marker: drupalMap.MarkerInterface): void

  /**
   * Removes a specified marker from the map.
   *
   * @param {drupalMap.MarkerInterface} marker - The marker object to be removed
   *   from the map.
   */
  removeMarker(marker: drupalMap.MarkerInterface): void

  /**
   * Sets a custom icon for the specified map marker.
   *
   * @param {Object} marker The marker object that the icon will be applied to.
   *   It should include properties and methods defined in both the base marker
   *   object and the ExtendedMarker interface.
   * @param {drupalMap.IconInterface} icon The icon object containing the data
   *   or configuration needed to render the marker icon.
   */
  setMarkerIcon(marker: any & drupalMap.Provider.ExtendedMarker, icon: drupalMap.IconInterface): void

  /**
   * Adds an event listener to a marker.
   *
   * The child classes should ensure that "this" inside the callback points to
   * the target marker.
   *
   * @param {any} marker - The marker object to add the event listener to.
   * @param {string} event - The event name to listen for.
   * @param {function} callback - The event handler function to be called when
   *   the event occurs.
   */
  addMarkerEventListener(marker: any, event: string, callback: (...args: any[]) => void): void

  /**
   * Determines whether the specified marker is currently visible.
   *
   * @param {any} marker - The marker object to check for visibility.
   *
   * @return {boolean} True if the marker is visible, otherwise false.
   */
  isMarkerVisible(marker: any): boolean

  /**
   * Set the content of the info window.
   *
   * Child classes may override this method if their map provider supports the
   * "info window" feature.
   *
   * @param {string|HTMLElement} content - The content to be set in the info window.
   *   If a string is provided, it can contain HTML markup. If an HTMLElement
   *   is provided, it is directly used as the content.
   */
  setInfoWindowContent(content: string | HTMLElement): void

  /**
   * Opens the info window.
   *
   * Child classes may override this method if their map provider supports the
   * "info window" feature.
   */
  openInfoWindow(anchor: any | drupalMap.MapPointInterface | drupalMap.Provider.ExtendedMarker): void

  /**
   * Closes the info window.
   *
   * Child classes may override this method if their map provider supports the
   * "info window" feature
   */
  closeInfoWindow(): void

  /**
   * Adds an event listener to the info window.
   *
   * Child classes may override this method if their map provider supports the
   * "info window" feature
   */
  addInfoWindowEventListener(event: string, callback: (...args: any[]) => void): void

  /**
   * Determines whether the map provider supports marker clusterer.
   *
   * Child classes may override this method if their map provider supports
   * marker clusterer.
   *
   * @return {boolean} True if Marker Clusterer is supported, otherwise false.
   */
  supportMarkerClusterer(): boolean

  /**
   * Initializes the marker clusterer instance to manage and display clusters of markers on a map.
   * Ensures efficient rendering and interaction with a large number of markers.
   */
  initMarkerClusterer(): void

  /**
   * Adds an array of markers to the clusterer for grouping on the map.
   *
   * @param {any[]} markers - An array of marker objects to be added to the clusterer.
   */
  addMarkersToClusterer(markers: any[]): void

  /**
   * Removes all markers from the associated marker clusterer.
   */
  clearMarkersFromClusterer(): void

}

interface ProviderMapStaticInterface {

  /**
   * Creates a Map object.
   *
   * @constructor
   *
   * @param {HTMLElement} element - The HTML element where the map should be
   *   displayed.
   * @param {ProviderMapSettings|null} settings - The map options.
   */
  new(element: HTMLElement, settings: ProviderMapSettings): ProviderMapInterface

  /**
   * Checks if all the JavaScript libraries for the provider are ready.
   *
   * Child classes should override this method.
   *
   * @return {boolean} Returns true if the provider is ready, false otherwise.
   */
  providerIsReady(): boolean

  /**
   * Add a new subscriber to the observable or event source.
   *
   * @param {drupalMap.SubscriberInterface} subscriber - An object implementing
   *   the SubscriberInterface. This object will receive updates or
   *   notifications from the source.
   */
  subscribe(subscriber: drupalMap.SubscriberInterface): void

  /**
   * Initializes and performs setup based on the provided context.
   *
   * @param {Document|DocumentFragment|Element} [context] - The context in which
   *   the initialization occurs. Defaults to the global document if not
   *   provided.
   */
  init(context?: Document|DocumentFragment|Element): void

  /**
   * Loads an external script asynchronously and returns a promise that resolves
   * when the script has been successfully loaded.
   *
   * @param {string} src - The source URL of the external script to load.
   *
   * @return {Promise<boolean>} A promise that resolves with `true` if the
   *   script is successfully loaded, or rejects if an error occurs during the
   *   loading process.
   */
  loadScriptAsync(src: string): Promise<boolean>

}

export interface ProviderMapSettings {

  /**
   * Specific settings related to a provider.
   */
  provider?: any

  infoWindowOptions?: any

  center?: {
    lat: number
    lng: number
  }

  style?: any

  clusterOptions?: any

  /**
   * The initial zoom level.
   */
  initialZoom?: number

  /**
   * The minimum zoom level.
   */
  minZoom?: number

  /**
   * The maximum zoom level.
   */
  maxZoom?: number

  /**
   * Whether to enable zooming using the scroll wheel.
   */
  scrollWheel?: boolean

}

export interface ModuleMapInterface {

  /**
   * The map instance built by the map provider.
   */
  mapObject: ProviderMapInterface

  /**
   * The current active marker instance.
   */
  markerActive?: drupalMap.MarkerInterface

  /**
   * The root element of the map.
   *
   * @return {HTMLElement}
   */
  getRoot(): HTMLElement

  /**
   * Retrieves the current Drupal map options.
   *
   * @return {drupalMap.Module.MapOptions} The currently set options for the Drupal map.
   */
  getOptions(): drupalMap.Module.MapOptions

  /**
   * Retrieves the value of an option by its key. If the key does not exist, returns the provided default value.
   *
   * @param {string} key - The key of the option to retrieve.
   * @param {*} [defaultValue=undefined] - The default value to return if the key does not exist.
   * @return {*} The value of the option if the key exists, otherwise the default value.
   */
  getOption(key: string, defaultValue?: any): any

  /**
   * Removes a marker identified by the specified key.
   *
   * @param {any} key - The unique identifier of the marker to be removed.
   */
  removeMarker(key: any): void

  /**
   * Disables the currently active marker, if any.
   */
  disableActiveMarker(): void

  /**
   * Adds an event listener for a specified event type.
   *
   * @param {string} type - The name of the event to listen for.
   * @param {(event: MapEvent) => void} listener - A callback function that will
   *   be invoked when the event is triggered.
   */
  addEventListener(type: string, listener: ((event: drupalMap.Module.MapEvent) => void)): void

  /**
   * Dispatches a custom event of the specified type with optional data.
   *
   * @param {string} type - The type of the event to be dispatched.
   * @param {{ [key: string]: any }} [data] - Optional additional data to include in the event details.
   */
  dispatchEvent(type: string, data?: { [key: string]: any }): void

  /**
   * Updates the icons of all markers on the map, if applicable.
   */
  rebuildIcons(): void

  /**
   * Rebuilds the cluster of markers on the map, if applicable.
   *
   * All map providers may not support clustering.
   */
  rebuildCluster(): void

}

export interface ModuleMapStaticInterface {

  new(root: HTMLElement, mapObject: ProviderMapInterface, options: drupalMap.Module.MapOptions): ModuleMapInterface

  /**
   * Prepare the map options.
   *
   * @param {Partial<drupalMap.Module.MapOptions>} options
   */
  buildOptions(options?: Partial<drupalMap.Module.MapOptions>): drupalMap.Module.MapOptions

}

interface LocationMapInterface extends ModuleMapInterface {

  locationListContainer: HTMLElement
  markerList: MarkerList<HTMLElement, drupalMap.LocationInterface>
  markerActive?: drupalMap.LocationInterface

  /**
   * Extracts the location information from a location-related event.
   *
   * @param {Event} event - The event object triggered from an interaction with
   *   a location .
   *
   * @return {drupalMap.LocationInterface|undefined} - The location data
   *   retrieved from the event, or undefined if not found.
   */
  getLocationFromLocationEvent(event: Event): drupalMap.LocationInterface|undefined

  /**
   * Enables the provided locations.
   *
   * @param {string[]} locationIds - An array of location IDs to enable.
   */
  setEnabledLocations(locationIds: string[]): void

}

interface SimpleMapInterface extends ModuleMapInterface {

  markerList: MarkerList<number, drupalMap.MarkerInterface>
  markerActive?: drupalMap.MarkerInterface

}

interface SimpleMapStaticInterface extends ModuleMapStaticInterface {

  new(root: HTMLElement, mapObject: ProviderMapInterface, options: drupalMap.Module.SimpleMapOptions): SimpleMapInterface

}

interface LocationMapStaticInterface extends ModuleMapStaticInterface {

  new(root: HTMLElement, mapObject: ProviderMapInterface, options: drupalMap.Module.LocationMapOptions): LocationMapInterface

  /**
   * Parses the location element data.
   *
   * @param {any} element - The location element to parse.
   * @param {string} dataAttribute - The data attribute name to look for data.
   * @param {boolean} throwError - Whether to throw an error if the data is
   *   invalid. Default is false.
   *
   * @returns {drupalMap.LocationData|undefined} - The parsed LocationData
   *   object or undefined if the data is invalid.
   * @throws {Error} - Error message if throwError is true and the data is
   *   missing, malformed, or invalid.
   */
  parseLocationData(element: any, dataAttribute: string, throwError?: boolean): drupalMap.LocationData|undefined

}

interface MarkerDataCoordinatesString {
  coordinates: string
  lat?: never
  lng?: never
}

interface MarkerDataCoordinatesLatLon {
  coordinates?: never
  lat: number|string
  lng: number|string
}

export interface DrupalMapInterface {

  /**
   * Create a subscriber to initialize maps of a given type.
   *
   * @class Subscriber
   * @constructor
   */
  Subscriber: typeof Subscriber

  /**
   * Abstract base class for a Map of a specific provider.
   *
   * Defines a list of methods that should be implemented by child classes. It
   * is used as an abstraction to ensure compatibility between tha various map
   * providers.
   *
   * @class ProviderMapBase
   */
  Map: typeof ProviderMapBase

  /**
   * Represents a point on a map.
   *
   * @cmass MapPoint
   *
   * @property {number} latitude - The latitude of the map point.
   * @property {number} longitude - The longitude of the map point.
   */
  MapPoint?: typeof MapPoint

  /**
   * Represents a marker.
   *
   * A marker is a point on the map with some extra data and additional
   * features.
   *
   * @class Location
   */
  Marker?: typeof Marker

  /**
   * Represents a location.
   *
   * A location is an HTML element with JSON data stored in a predefined data
   * attribute, containing at least the properties:
   * - id: A unique identifier.
   * - coordinates: The location coordinates in a format supported by the
   *   MapPoint class constructor.
   *
   * The HTML content of the location element is used by the LocationMap module
   * for the information window associated with the marker.
   *
   * @class Location
   */
  Location?: typeof Location

  /**
   * Represents a MapIcon.
   *
   * The icon can be created from a remote URL or a base64 encoded string.
   *
   * @class MapIcon
   */
  MapIcon?: typeof MapIcon

  /**
   * Stores all initialized map instances, keyed by their root element.
   *
   * @type {Map<HTMLElement, V extends ModuleMapInterface>}
   */
  instances?: Map<HTMLElement, ModuleMapInterface>

  /**
   * A provider is a specialized class implementing the BaseMap class.
   *
   * When applicable, a provider ca also expose a Clusterer constructor to
   * create groups of markers.
   */
  providers?: {
    [key: string]: {
      Map: ProviderMapStaticInterface
      MarkerClusterer?: new(...args: any[]) => any
    }
  }

  /**
   * A module is a type of map with specific features.
   *
   * The native modules are:
   * - SimpleMap: A simple map displaying markers.
   * - LocationMap: An advanced map based on a list of locations, which are
   *   parsed on initialization to generate the markers displayed on the map.
   *   The module provides:
   *   - Display if an info window.
   *   - Auto-zoom when a location is clicked.
   *   - Grouping of markers using Clusterer, when applicable.
   */
  modules?: {

    [key: string]: ModuleMapStaticInterface
    SimpleMap?: SimpleMapStaticInterface
    LocationMap?: LocationMapStaticInterface

  }

  /**
   * Retrieves the root element from the given settings.
   *
   * @param {Object} settings - The settings for the map.
   * @param {string} settings.selector - The CSS selector of the root element.
   *
   * @return {HTMLElement|undefined} - The root element of the map, or undefined if not found.
   */
  getMapRootElement(settings: any): HTMLElement|undefined

  /**
   * Retrieves the map module asynchronously.
   *
   * @param {HTMLElement} mapRoot - The root element where the map will be mounted.
   *
   * @return {Promise<ProviderMapInterface>} - A Promise that resolves to the ProviderMap instance.
   */
  getMapModuleAsync(mapRoot: HTMLElement): Promise<ProviderMapInterface>

}

export interface BaseMapsSettings {
  selector: string
  provider: string
  options?: Partial<drupalMap.Module.MapOptions>
  initialized?: boolean
}

export interface SimpleMapsSettings extends BaseMapsSettings {
  markers?: drupalMap.Module.MarkerDefinition[]
}

export interface LocationMapsSettings extends BaseMapsSettings {
  enabledLocations?: string[]
}

declare namespace drupalMap {

  export type SubscriberInterface = Subscriber

  export type SubscriberCallback = (namespace: string, element: HTMLElement, options: drupalMap.Module.MapOptions, mapSettings: BaseMapsSettings) => boolean|null

  export interface LatLng {
    /** Latitude in degrees. */
    lat: number
    /** Longitude in degrees. */
    lng: number
  }

  export type MapPointInterface = MapPoint

  export type MarkerDataOptions = {
    [key: string]: any
    infoWindow?: string
  }

  export type MarkerData = (MarkerDataCoordinatesString | MarkerDataCoordinatesLatLon) & {
    [key: string]: any
    id?: string|number
    options: MarkerDataOptions
  }

  export type MarkerInterface = Marker

  export type LocationData = drupalMap.MarkerData & {
    id?: string
    title?: string
    address?: string
    city?: string
    postalCode?: string
  }

  export type LocationInterface = Location

  export type IconInterface = MapIcon

  namespace Module {

    export interface MarkerDefinition {
      [key: string]: any
      lat: number
      lng: number
      options: {
        [key: string]: any
      }
    }

    export interface MapSelectors {
      map?: string
      [key: string]: string
    }

    export interface MapOptions {
      [key: string]: any
      map?: ProviderMapSettings
      markerOptions?: {
        iconCallback?: (event: drupalMap.Module.MapMarkerEvent) => void
      }
    }

    export interface SimpleMapOptions extends MapOptions {
      markers?: MarkerDefinition[]
    }

    export interface LocationMapOptions extends MapOptions {
      hideNotVisibleLocations?: boolean
      selectors: MapSelectors
      zoomOnLocationClick?: number
      location?: {
        dataAttribute?: string
      }
    }

    export type MapEventData = {
      [key: string]: any
      map: ModuleMapInterface
    }

    export interface MapEvent extends CustomEvent<MapEventData> {}

    export interface MapInfoWindowEvent extends CustomEvent<MapEventData & {
      marker: drupalMap.MarkerInterface
      element: HTMLElement
    }> {}

    export interface MapMarkerEvent {
      marker: Marker
      markerIsActive: boolean
      type: string
    }

  }

  namespace Provider {

    export interface ExtendedMarker {
      a12sMarker?: drupalMap.MarkerInterface
    }

  }

  namespace Icon {

    export interface Size {
      height: number
      width: number
    }

    export interface Position {
      x: number
      y: number
    }

    export interface Options {
      size?: drupalMap.Icon.Size
      anchor?: drupalMap.Icon.Position
      origin?: drupalMap.Icon.Position
    }

  }

}

export interface a12sLocationsMapUpdateCommand extends IAjaxCommand {
  (
    ajax: IAjax,
    response: a12sLocationsMapUpdateResponse,
    status: number
  ): void
}

interface a12sLocationsMapUpdateResponse {
  data: any
  enabledLocations: string[]
  selector: string
}
