import ModuleMapBase from "./modules/ModuleMapBase";
import {ModuleMapInterface} from "../types";

/**
 * Stores the initialized map instances.
 */
export default class MapInstances extends Map<HTMLElement, ModuleMapInterface> {

  /**
   * Mutation observer used to keep the list of map instances up to date.
   *
   * @protected {MutationObserver}
   */
  protected readonly deletionObserver: MutationObserver;

  /**
   * The mutation observer options.
   *
   * @protected MutationObserverInit
   */
  protected readonly observerOptions: MutationObserverInit = {childList: true, subtree: true};

  /**
   * Override the Map constructor to initialize the mutation observer.
   *
   * @param {[HTMLElement, ModuleMapInterface][]|null} entries
   */
  constructor(entries?: (readonly [HTMLElement, ModuleMapInterface])[] | null) {
    super(entries);
    this.deletionObserver = new MutationObserver(this.mapDeletionObserver.bind(this));
  }

  /**
   * Override the Map method to start the mutation observer.
   *
   * @param {HTMLElement} key
   *   The root element.
   * @param {ModuleMapInterface} value
   *   The map instance.
   */
  set(key: HTMLElement, value: ModuleMapInterface): this {
    super.set(key, value);
    this.deletionObserver.observe(document.body, this.observerOptions);
    return this;
  }

  /**
   * Override the Map method to stop the mutation observer if no more necessary.
   *
   * @param {HTMLElement} key
   *   The root element.
   */
  delete(key: HTMLElement): boolean {
    const result = super.delete(key);

    if (!this.size) {
      this.deletionObserver.disconnect();
    }

    return result;
  }

  /**
   * Callback for the mutation observer.
   *
   * @param {MutationRecord[]} mutationList
   */
  mapDeletionObserver(mutationList: MutationRecord[]/*, observer: MutationObserver*/): void {
    for (const mutation of mutationList) {
      if (mutation.type === 'childList' && mutation.removedNodes.length) {
        const deletedNodes = Array.from(mutation.removedNodes);

        this.forEach((map, element) => {
          if (deletedNodes.find((node) => element === node || node.contains(element))) {
            this.delete(element);
          }
        });
      }
    }
  }

}
