import {
  drupalMap,
  ModuleMapInterface
} from "../types";

export class Marker {

  readonly mapPoint: drupalMap.MapPointInterface;

  protected _marker: any;

  protected _enabled: boolean = true;

  constructor(readonly map: ModuleMapInterface, protected _data: drupalMap.MarkerData) {
    this._data.options = this._data.options || {};

    if ('coordinates' in this._data) {
      this.mapPoint = new window.DrupalMap.MapPoint(this.data.coordinates);
    }
    else if ('lat' in this._data && 'lng' in this._data) {
      this.mapPoint = new window.DrupalMap.MapPoint(Number(this._data.lat.toString()), Number(this._data.lng.toString()));
    }
    else {
      // Force throwing error from MapPoint constructor.
      this.mapPoint = new window.DrupalMap.MapPoint(null);
    }

    this.map.dispatchEvent('alterA12sDrupalMapMarker', {data: this._data});

    if ('enabled' in this._data) {
      this._enabled = !!this._data.enabled;
    }

    this._marker = this.map.mapObject.addMarker(this, this._enabled);
  }

  /**
   * @return {any} The marker built by the map provider (Google, Baidu...).
   */
  get marker(): any {
    return this._marker;
  }

  get enabled(): boolean {
    return this._enabled;
  }

  set enabled(enabled) {
    if (enabled !== this._enabled) {
      this._enabled = enabled;
      enabled ? this.map.mapObject.enableMarker(this) : this.map.mapObject.disableMarker(this);
    }
  }

  get data(): drupalMap.MarkerData {
    return this._data;
  }

  getId(): string|undefined {
    if ('id' in this.data) {
      const id = typeof this.data.id === 'number' ? this.data.id.toString() : this.data.id;
      return id !== '' ? id : undefined;
    }
  }

  get options(): drupalMap.MarkerDataOptions {
    return this._data.options;
  }

  getOption(key: string): any {
    return key in this.options ? this.options[key] : undefined;
  }

  public addMarkerEventListener(event: string, callback: (this: any & drupalMap.Provider.ExtendedMarker, ...args: any[]) => void): void {
    this.map.mapObject.addMarkerEventListener(this.marker, event, function (...args) {
      // "this" should point to the target marker. If so, we can forward to the
      // callback function.
      if (this.hasOwnProperty('a12sMarker')) {
        const marker = this.a12sMarker;

        if (marker) {
          callback.apply(this, args);
          return;
        }
      }

      console.warn(`Trying to use a location callback for event "${event}" on an object which is not an extended map marker.`);
    });
  }

  public setMarkerIcon(icon: drupalMap.IconInterface): void {
    this.map.mapObject.setMarkerIcon(this.marker, icon);
  }

  /**
   *
   * Event types:
   * - mouseOverLocationElement
   * - mouseOutLocationElement
   * - disableActiveMarker
   * - mouseOverMarker
   * - mouseOutMarker
   * - enableActiveMarker
   *
   * @todo replace this by a publisher/subscriber system.
   */
  updateState(event: Partial<drupalMap.Module.MapMarkerEvent>): void {
    const map = this.map as ModuleMapInterface;

    if (event.type === 'disableActiveMarker') {
      this.map.markerActive = undefined;
      map.mapObject.closeInfoWindow();
      this.map.dispatchEvent('closeInfoWindow', {
        marker: this,
      });
    }

    if (event.type === 'enableActiveMarker') {
      map.disableActiveMarker();
      map.markerActive = this;
      this.displayInfoWindow();
    }

    this.updateIcon(event);
  }

  updateIcon(event: Partial<drupalMap.Module.MapMarkerEvent>): void {
    const iconCallback = this.map.getOption('markerOptions')?.iconCallback || null;

    if (typeof iconCallback === 'function') {
      const iconEvent: drupalMap.Module.MapMarkerEvent = Object.assign({
        type: 'unknown',
        markerIsActive: false
      }, event, {marker: this});
      iconCallback(iconEvent);
    }
    else {
      const icon = this.getOption('icon');

      if (icon && 'url' in icon && typeof icon.url === 'string') {
        const mapIcon = new window.DrupalMap.MapIcon(icon.url, icon.options || {});
        this.setMarkerIcon(mapIcon);
      }
    }
  }

  protected getInfoWindowHtml(): HTMLElement {
    const infoWindow: string|undefined = this.getOption('infoWindow');

    if (infoWindow) {
      // Convert to HTML and sanitize it.
      const parser = new DOMParser();
      const body = parser.parseFromString(infoWindow, 'text/html').body;
      const content = document.createElement('div');
      content.classList.add('marker-info-window');
      content.innerHTML = body.innerHTML;
      this.cleanElements(content);
      return content;
    }
  }

  displayInfoWindow(): void {
    const infoWindow = this.getInfoWindowHtml();

    if (infoWindow) {
      const providerMap = this.map.mapObject;
      providerMap.setInfoWindowContent(infoWindow);
      providerMap.openInfoWindow(this);

      this.map.dispatchEvent('openInfoWindow', {
        marker: this,
        element: infoWindow,
      });
    }
  }

  cleanElements(html: HTMLElement): void {
    if (typeof html.remove === 'function') {
      html.remove();
      return;
    }

    /* @ts-ignore TS2488 */
    for (let node of html.children) {
      if (node.nodeName === 'SCRIPT') {
        node.remove();
        continue;
      }

      for (let {name, value} of node.attributes) {
        value = value.replace(/\s+/g, '').toLowerCase();

        if (['src', 'href', 'xlink:href'].includes(name)) {
          if (value.includes('javascript:') || value.includes('data:text/html')) {
            node.removeAttribute(name);
            continue;
          }
        }

        if (name.startsWith('on')) {
          node.removeAttribute(name);
        }
      }

      this.cleanElements(node as HTMLElement);
    }
  }

}
