import {
  drupalMap,
  ModuleMapInterface,
  ProviderMapInterface
} from "../../types";
import MarkerList from "../MarkerList";

/**
 * Base class for modules.
 *
 * Its purpose is to provide base methods to create a map with essential
 * features for managing map options, dispatching events, and handling markers
 * or clusters.
 */
export default abstract class ModuleMapBase<K, V extends drupalMap.MarkerInterface> implements ModuleMapInterface {

  readonly markerList: MarkerList<K, V>;

  public markerActive?: V;

  /**
   * Construct a new module map.
   *
   * @param {HTMLElement} root The root HTML element, defining the map.
   * @param {ProviderMapInterface} mapObject The map object, depending on the provider ("google", "baidu"...).
   * @param {Partial<drupalMap.Module.MapOptions>} options The list of options for the current instance.
   */
  protected constructor(protected readonly root: HTMLElement, readonly mapObject: ProviderMapInterface, protected options: Partial<drupalMap.Module.MapOptions>) {
    this.markerList = new MarkerList();
  }

  /**
   * @see ModuleMapStaticInterface.buildOptions
   */
  public static buildOptions(options?: Partial<drupalMap.Module.MapOptions>): drupalMap.Module.MapOptions {
    options = options || {};
    options.selectors = options.selectors || {};
    return options as drupalMap.Module.MapOptions;
  }

  getRoot(): HTMLElement {
    return this.root;
  }

  getOptions(): drupalMap.Module.MapOptions {
    return this.options;
  }

  getOption(name: string, defaultValue: any = undefined): any {
    return this.options.hasOwnProperty(name) ? this.options[name] : defaultValue;
  }

  addMarker(key: K, marker: V): void {
    const map = this;
    this.markerList.set(key, marker);

    marker.addMarkerEventListener('mouseover', function(this: any & drupalMap.Provider.ExtendedMarker) {
      marker.updateState({ markerIsActive: map.markerActive === this.a12sMarker, type: 'mouseOverMarker' });
    });

    marker.addMarkerEventListener('mouseout', function(this: any & drupalMap.Provider.ExtendedMarker) {
      if (map.markerActive !== this.a12sMarker) {
        marker.updateState({ markerIsActive: map.markerActive === this.a12sMarker, type: 'mouseOutMarker' });
      }
    });

    marker.addMarkerEventListener('click', function(this: any & drupalMap.Provider.ExtendedMarker) {
      if (map.markerActive === this.a12sMarker) {
        map.disableActiveMarker();
      }
      else {
        marker.updateState({ markerIsActive: true, type: 'enableActiveMarker' });
      }
    });

    this.mapObject.addInfoWindowEventListener('close', () => {
      if (this.markerActive) {
        this.disableActiveMarker();
      }
    });

    marker.updateState({
      type: 'createMarker',
      markerIsActive: false
    });
  }

  removeMarker(key: K): void {
    const marker = this.markerList.get(key);

    if (marker) {
      this.markerList.delete(key);
      marker.marker.remove();
    }
  }

  disableActiveMarker(): void {
    if (this.markerActive) {
      this.markerActive.updateState({
        markerIsActive: false,
        type: 'disableActiveMarker'
      });
    }
  }

  addEventListener(type: string, listener: ((event: drupalMap.Module.MapEvent) => void)): void {
    this.root.addEventListener(type, listener);
  }

  dispatchEvent(type: string, data?: { [key: string]: any }): void {
    const details = data || {};
    details.map = this;

    const event = new CustomEvent(type, {
      detail: details
    });

    this.root.dispatchEvent(event);
  }

  rebuildIcons(): void {
    this.markerList.forEach(marker => {
      marker.updateState({
        type: 'createMarker',
        markerIsActive: false
      });
    });
  }

  rebuildCluster() {
    this.mapObject.clearMarkersFromClusterer();
    this.mapObject.addMarkersToClusterer(this.markerList.getMarkers());
  }

}
