import {
  drupalMap,
  LocationMapInterface,
  ProviderMapInterface
} from "../../../types";
import MarkerList from "../../MarkerList";
import ModuleMapBase from "../ModuleMapBase";

export default class LocationMap extends ModuleMapBase<HTMLElement, drupalMap.LocationInterface> implements LocationMapInterface {

  static readonly SELECTORS_DEFAULT: drupalMap.Module.MapSelectors = {
    map: '[data-a12s-locations-map]',
    locationList: '[data-a12s-locations-list]',
    locationElement: '[data-a12s-locations-point]',
  };

  readonly locationListContainer: HTMLElement;

  readonly markerList: MarkerList<HTMLElement, drupalMap.LocationInterface>;

  public markerActive?: drupalMap.LocationInterface;

  protected currentSearch: null | string;

  /**
   * Construct a new Store Locator map object.
   *
   * @param {HTMLElement} root The root HTML element, defining the map...
   * @param {ProviderMapInterface} mapObject The map object, depending on the provider ("google", "baidu"...).
   * @param {Partial<drupalMap.Module.LocationMapOptions>} options
   */
  constructor(protected readonly root: HTMLElement, readonly mapObject: ProviderMapInterface, protected options: Partial<drupalMap.Module.LocationMapOptions> = {}) {
    super(root, mapObject, options);
    options.hideNotVisibleLocations = options.hideNotVisibleLocations || false;
    options.zoomOnLocationClick = options.zoomOnLocationClick || undefined;

    this.currentSearch = null;
    this.locationListContainer = root.querySelector(options.selectors.locationList);

    if (this.locationListContainer) {
      this.locationListContainer.querySelectorAll(options.selectors.locationElement).forEach((element) => {
        try {
          const data = LocationMap.parseLocationData(element, this.options.location?.dataAttribute || null, true);
          // We can safely cast the "element" to HTMLElement, as the Location
          // class will throw an error if it is not of such type.
          const location = new window.DrupalMap.Location(element as HTMLElement, this, data);
          this.addMarker(element as HTMLElement, location);
        }
        catch (error) {
          element.remove();
          console.log(`Invalid location: ${error.message}`);
        }
      });

      this.locationListContainer.addEventListener('click', (event) => {
        const mapPoint = this.getLocationFromLocationEvent(event)?.mapPoint;

        if (mapPoint) {
          if (this.options.zoomOnLocationClick !== undefined) {
            this.mapObject.setZoom(this.options.zoomOnLocationClick);
          }

          this.mapObject.setCenter(mapPoint);
          this.disableActiveMarker();
          this.mapObject.closeInfoWindow();
        }
      });

      this.locationListContainer.addEventListener('mouseover', (event) => {
        const location = this.getLocationFromLocationEvent(event);

        location?.updateState({
          markerIsActive: this.markerActive === location,
          type: 'mouseOverLocationElement'
        });
      });

      this.locationListContainer.addEventListener('mouseout', (event) => {
        const location = this.getLocationFromLocationEvent(event);

        location?.updateState({
          markerIsActive: this.markerActive === location,
          type: 'mouseOutLocationElement'
        });
      });
    }

    if (options.hideNotVisibleLocations) {
      this.mapObject.addMapEventListener('bounds_changed', () => {
        this.markerList.forEach((location) => {
          if (location.enabled) {
            location.HTMLElement.hidden = !(this.mapObject.isMarkerVisible(location.marker));
          }
        });
      });
    }

    this.mapObject.initMarkerClusterer();
    this.mapObject.addMarkersToClusterer(this.markerList.getMarkers());
    this.dispatchEvent('initializedA12sLocationsMap');
  }

  public static buildOptions(options?: Partial<drupalMap.Module.LocationMapOptions>): drupalMap.Module.LocationMapOptions {
    options = options || {};
    options.selectors = Object.assign(this.SELECTORS_DEFAULT, options.selectors || {});
    options.location = Object.assign({ dataAttribute: window.DrupalMap.Location.LOCATION_DATA_ATTRIBUTE_DEFAULT }, options.location || {});
    return options as drupalMap.Module.LocationMapOptions;
  }

  getLocationFromLocationEvent(event: Event): drupalMap.LocationInterface|undefined {
    if (event.target && event.target instanceof Element) {
      const locationElement = event.target.closest(this.options.selectors.locationElement);

      if (locationElement instanceof HTMLElement) {
        return this.markerList.get(locationElement);
      }
    }
  }

  setEnabledLocations(locationIds: string[]): void {
    this.markerList.forEach((location) => {
      location.enabled = locationIds.includes(location.getId());
    });

    this.rebuildCluster();
    this.dispatchEvent('updateEnabledA12sLocations');
  }

  /**
   * @see LocationMapStaticInterface.parseLocationData
   */
  public static parseLocationData(element: any, dataAttribute: string = 'locationMarker', throwError: boolean = false): drupalMap.LocationData|undefined {
    if (!(element instanceof HTMLElement) || !(dataAttribute in element.dataset) || !element.dataset[dataAttribute]) {
      if (throwError) {
        throw new Error('Missing marker data for the location element.');
      }
    }
    else {
      let data;

      try {
        data = JSON.parse(element.dataset[dataAttribute]);
      }
      catch (e) {
        if (throwError) {
          throw new Error('Malformed JSON marker data for the element.');
        }
      }

      if (data && (typeof data.coordinates === 'undefined' || typeof data.id === 'undefined')) {
        if (throwError) {
          throw new Error('Invalid marker data for the element.');
        }

        data = undefined;
      }
      else if (typeof data.id === 'number') {
        data.id = data.id.toString();
      }

      return data;
    }
  }

}
