import {
  drupalMap,
  ProviderMapSettings,
  ProviderMapInterface
} from "../../types";
import MapPoint from "../MapPoint";

/**
 * Handle the initialization of a Map using Drupal settings.
 *
 * @class
 */
class Initializer {

  initCallbacks: drupalMap.SubscriberInterface[] = [];

  add(subscriber: drupalMap.SubscriberInterface) {
    this.initCallbacks.push(subscriber);
  }

  /**
   * Map init callback.
   *
   * This method may be called in different ways, depending on the map provider.
   *
   * For example, with Google, this method is defined as a callback when the
   * GMap script is loaded:
   * https://maps.googleapis.com/maps/api/js?key=...&callback=drupalMap.google.init:
   *
   * With Baidu, there is no such load callback, so we handle this by ourselves
   * to have a similar behavior.
   */
  init = (context?: Document|DocumentFragment|Element) => {
    this.initCallbacks.forEach((subscriber) => {
      subscriber.initAllMaps(context);
    });
  }

}

const initializer = new Initializer();

/**
 * A class representing an abstract map.
 *
 * This class is supposed to be extended by specialized code for a specific map
 * provider (Google, Baidu...).
 *
 * @class
 * @property {HTMLElement} element - The HTML element containing the map.
 * @property {any} map - The Map object (this depends on the map provider).
 */
export default abstract class ProviderMapBase implements ProviderMapInterface {

  readonly map: any;

  /**
   * Represents an information window typically used in mapping or UI applications.
   * The variable can hold any data type, and its contents are dependent on the implementation.
   * Commonly used to display temporary or contextual information to users.
   *
   * @type {*} Any type of data can be assigned to this variable.
   */
  protected infoWindow: any;

  /**
   * A variable to manage a cluster of markers, typically used in map-based applications to group
   * closely located geographical points into a single cluster for better visualization and performance.
   *
   * @type {any}
   * @default null
   */
  protected markerCluster: any = null;

  protected constructor(readonly element: HTMLElement, public settings: ProviderMapSettings = {}) {
    this.infoWindow = null;
    this.map = this.initMap(element);
  }

  /**
   * @see ProviderMapStaticInterface.providerIsReady
   */
  public static providerIsReady(): boolean {
    return false;
  }

  /**
   * Initializes a map on the provided HTML element.
   *
   * @param {HTMLElement} element - The HTML element on which the map will be
   *   initialized.
   * @return {any} The initialized map instance or a relevant value indicating
   *   the result of the initialization.
   */
  protected abstract initMap(element: HTMLElement): any;

  public abstract setCenter(point: drupalMap.MapPointInterface): void;

  public abstract getZoom(): number;

  public abstract setZoom(zoom: number): void;

  public abstract addMapEventListener(event: string, callback: (...args: any[]) => void): void;

  public abstract setBoundsFromMarkers(markers: any[]): void;

  public abstract setBoundsFromMapPoints(points: drupalMap.MapPointInterface[]): void;

  public addMarker(marker: drupalMap.MarkerInterface|drupalMap.MapPointInterface, enabled?: boolean): any {
    const isMapPoint = marker instanceof MapPoint;
    const lat = isMapPoint ? marker.lat : marker.mapPoint.lat;
    const lng = isMapPoint ? marker.lng : marker.mapPoint.lng;
    let title = 'data' in marker && 'title' in marker.data ? marker.data.title : `Point (${lat}, ${lng})`;
    const markerInstance = this.createMarker(lat, lng, enabled, title);

    if (!isMapPoint) {
      markerInstance.a12sMarker = marker;
    }

    return markerInstance;
  }

  /**
   * Creates a marker on the map with the specified latitude, longitude, and
   * optional properties.
   *
   * @param {number} lat - The latitude where the marker will be placed.
   * @param {number} lng - The longitude where the marker will be placed.
   * @param {boolean} [enabled] - An optional flag indicating whether the marker is enabled. Defaults to false if not provided.
   * @param {string} [title] - An optional title for the marker.
   *
   * @return {any} The created marker object.
   */
  protected abstract createMarker(lat: number, lng: number, enabled?: boolean, title?: string): any;

  public abstract enableMarker(marker: drupalMap.MarkerInterface): void;

  public abstract disableMarker(marker: drupalMap.MarkerInterface): void;

  public abstract removeMarker(marker: drupalMap.MarkerInterface): void;

  public abstract setMarkerIcon(marker: any & drupalMap.Provider.ExtendedMarker, icon: drupalMap.IconInterface): void;

  public abstract addMarkerEventListener(marker: any, event: string, callback: (...args: any[]) => void): void;

  public abstract isMarkerVisible(marker: any): boolean;

  public setInfoWindowContent(content: string | HTMLElement): void {}

  public openInfoWindow(anchor: any | drupalMap.MapPointInterface | drupalMap.Provider.ExtendedMarker): void {}

  public closeInfoWindow(): void {}

  public addInfoWindowEventListener(event: string, callback: (...args: any[]) => void): void {}

  public supportMarkerClusterer(): boolean {
    return false;
  }

  public initMarkerClusterer(): void {}

  public addMarkersToClusterer(markers: any[]): void {}

  public clearMarkersFromClusterer(): void {}

  /**
   * @see ProviderMapStaticInterface.subscribe
   */
  public static subscribe(subscriber: drupalMap.SubscriberInterface) {
    initializer.add(subscriber);
  }

  /**
   * @see ProviderMapStaticInterface.init
   */
  public static init(context?: Document|DocumentFragment|Element) {
    initializer.init(context);
  }

  /**
   * @see ProviderMapStaticInterface.loadScriptAsync
   */
  public static async loadScriptAsync(src: string): Promise<boolean> {
    const script = document.createElement('script');
    script.src = src;
    script.type = 'text/javascript';

    return new Promise((resolve, reject) => {
      script.onload = () => {
        resolve(true);
      };

      script.onerror = () => {
        reject(`Loading script "${src}" failed.`);
      }

      document.getElementsByTagName('head')[0].appendChild(script);
    });
  }

}
