<?php

namespace Drupal\a12s_locations\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\ConfigTarget;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Form\ToConfig;

/**
 * Location map settings form.
 */
class LocationsMapSettingsForm extends ConfigFormBase {

  const A12S_LOCATIONS_SETTINGS = 'a12s_locations.settings';

  /**
   * @inheritDoc
   */
  protected function getEditableConfigNames(): array {
    return [self::A12S_LOCATIONS_SETTINGS];
  }

  /**
   * @inheritDoc
   */
  public function getFormId(): string {
    return 'a12s_locations_settings_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $form = parent::buildForm($form, $form_state);
    $config = $this->config(self::A12S_LOCATIONS_SETTINGS);

    // Google Map API key.
    $form['google_map_api_key'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Google Map API key'),
      '#config_target' => self::A12S_LOCATIONS_SETTINGS . ':google_map_api_key',
    ];

    $form['google_map_api_places'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable Google Maps Places API'),
      '#default_value' => $config->get('google_map_api_places'),
      '#description' => $this->t('Enable Google Maps Places API to allow users to search for locations.'),
      '#config_target' => new ConfigTarget(
        self::A12S_LOCATIONS_SETTINGS,
        'google_map_api_places',
        toConfig: static::class . '::processGoogleMapPlacesApi'
      ),
      '#states' => [
        'visible' => [
          ':input[name="google_map_api_key"]' => ['!empty' => TRUE],
        ],
      ],
    ];

    // Baidu Map API key.
    $form['baidu_map_api_key'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Baidu Map API key'),
      '#config_target' => self::A12S_LOCATIONS_SETTINGS . ':baidu_map_api_key',
    ];

    return $form;
  }

  /**
   * Transformation callback for the google_map_api_places config value.
   *
   * @param  $google_map_api_places
   *   The config value to transform.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *
   * @return \Drupal\Core\Form\ToConfig|bool
   */
  public static function processGoogleMapPlacesApi( $google_map_api_places, FormStateInterface $form_state): ToConfig|bool {
    if (!trim($form_state->getValue('google_map_api_key'))) {
      return ToConfig::DeleteKey;
    }

    return (bool) $google_map_api_places;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    parent::submitForm($form, $form_state);
    \Drupal::service('cache_tags.invalidator')->invalidateTags(['library_info']);
    $this->messenger()->addStatus($this->t('Library definitions cache has been cleared.'));
  }

}
