<?php

namespace Drupal\a12s_maps_sync\Plugin\MappingHandler;

use Drupal\a12s_maps_sync\Converter\Mapping;
use Drupal\a12s_maps_sync\Entity\ConverterInterface;
use Drupal\a12s_maps_sync\Exception\MapsException;
use Drupal\a12s_maps_sync\Maps\BaseInterface;
use Drupal\a12s_maps_sync\Maps\MapsObject;
use Drupal\a12s_maps_sync\Plugin\MappingHandlerPluginBase;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Field\BaseFieldDefinition;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Language\LanguageInterface;

/**
 * Plugin implementation of the a12s_maps_sync_mapping_handler.
 *
 * @MappingHandler(
 *   id = "entity_reference",
 *   label = @Translation("Entity Reference"),
 *   description = @Translation("Entity Reference.")
 * )
 */
class EntityReferenceHandler extends MappingHandlerPluginBase {

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
      'source_converter' => NULL,
      'converters' => NULL,
      'entity_type' => NULL,
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form['converters'] = [
      '#type' => 'select',
      '#multiple' => TRUE,
      '#title' => $this->t('From converter'),
      '#empty_option' => $this->t('- Select -'),
      '#options' => $this->getMappingConverterOptions(),
      '#default_value' => $this->getConfiguration()['converters'],
      '#description' => $this->t('The converter used to map the related objects / medias.'),
    ];

    $form['entity_type'] = [
      '#type' => 'select',
      '#title' => $this->t('Entity type'),
      '#empty_option' => $this->t('- Use default -'),
      '#options' => $this->getMappingEntityTypeOptions(),
      '#default_value' => $this->getConfiguration()['entity_type'] ?? NULL,
      '#description' => $this->t('The target entity type. "- Use default -" will use the entity type defined in the "From converter", and is sufficient in most cases.'),
    ];

    return parent::buildConfigurationForm($form, $form_state);
  }

  /**
   * @return array
   */
  public function getMappingConverterOptions(): array {
    /** @var ConverterInterface $converter */
    $converter = $this->getConfiguration()['source_converter'];

    $options = [];
    if ($converter) {
      foreach ($converter->getProfile()->getConverters() as $targetConverter) {
        $options[$targetConverter->id()] = $targetConverter->label();
      }
    }

    return $options;
  }

  /**
   * @return array
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   */
  public function getMappingEntityTypeOptions(): array {
    $entity_types = array_filter(\Drupal::config('a12s_maps_sync.settings')->get('allowed_entity_types'));

    $options = [];
    foreach ($entity_types as $entity_type_id) {
      $entity_type = \Drupal::entityTypeManager()->getDefinition($entity_type_id);
      $options[$entity_type->id()] = $entity_type->getLabel();
    }
    asort($options);

    return $options;
  }

  /**
   * {@inheritdoc}
   */
  public function mapData(EntityInterface $entity, BaseInterface $object, Mapping $mapping, int $mapsLanguage, ?LanguageInterface $language = NULL) {
    $source = $mapping->getSource();
    $targetFieldName = $mapping->getTarget();

    $values = [];

    // Delete the previous value.
    if (!$mapping->isAppend()) {
      $entity->{$mapping->getTarget()} = [];
    }

    for ($i = 0; $i < $object->getCountValues($source); $i++) {
      foreach ($mapping->getConverters() as $targetConverter) {
        // We need to find the entity in Drupal.
        /** @var \Drupal\Core\Entity\EntityFieldManager $entityFieldManager */
        $entityFieldManager = \Drupal::service('entity_field.manager');
        $fields = $entityFieldManager->getFieldStorageDefinitions($entity->getEntityType()
          ->id());

        // Ensure that we have a field name configured in the mapping.
        if (empty($targetFieldName)) {
          \Drupal::logger('a12s_maps_sync')
            ->error('No target field name defined for target converter %converter', ['%converter' => $targetConverter->id()]);
          return FALSE;
        }

        if (empty($fields[$targetFieldName])) {
          \Drupal::logger('a12s_maps_sync')
            ->error('The field %field_name does not exist.', ['%field_name' => $targetFieldName]);
          return FALSE;
        }

        $target_entity = $this->getEntityFromGid($targetConverter, $object, $mapping, $i);

        if (!empty($target_entity)) {
          // Set the field value.
          $value = $this->getValue($target_entity);

          if ($this->isMultiple($entity, $mapping)) {
            $values[] = $value;
          }
          else {
            $values = $value;
          }
        }
        else {
          // Ultra special case for parents.
          if ($targetFieldName === 'parent') {
            // Set to root.
            $entity->get($targetFieldName)->target_id = 0;
          }
        }
      }
    }

    if (!$entity->get($targetFieldName)->isEmpty() && $mapping->isAppend()) {
      $currentValues = [];
      foreach ($entity->get($targetFieldName) as $targetEntity) {
        if ($targetEntity->entity !== NULL) {
          $currentValues[] = $targetEntity->entity->id();
        }
      }

      $values = array_unique(array_merge($currentValues, $values));
    }

    $entity->set($targetFieldName, $values);

    return $values;
  }

  /**
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *
   * @return int|string|null
   */
  protected function getValue(EntityInterface $entity) {
    return $entity->id();
  }

  /**
   * @param array $definition
   * @param array $array
   *
   * @return false|string
   */
  protected function getGidFromArray(array $definition, array $array): false|string {
    $gid = [];

    foreach ($definition as $item) {
      $exploded = explode(':', $item);

      // If the element begins with "const:", it is a constant, and we don't
      // have to get any value...
      if (count($exploded) === 2 && $exploded[0] === 'const') {
        $gid[] = $exploded[1];
      }
      else {
        if (empty($array[$item])) {
          return FALSE;
        }

        $gid[] = $array[$item];
      }
    }

    return implode('-', $gid);
  }

  /**
   * {@inheritdoc}
   */
  public function getObjectValue(ConverterInterface $converter, BaseInterface $object, Mapping $mapping, int $i) {
    $source = $mapping->getSource();

    if ($converter->getMapsType() === 'library') {
      $attribute = $object->getAttribute($source);
      return $attribute->getData()[$i]['idattribute_library'];
    }
    else {
      return $object->get($source, NULL, $i);
    }
  }

  /**
   * @param \Drupal\a12s_maps_sync\Entity\ConverterInterface $converter
   * @param \Drupal\a12s_maps_sync\Maps\BaseInterface $object
   * @param \Drupal\a12s_maps_sync\Converter\Mapping $mapping
   * @param int $i
   *
   * @return \Drupal\Core\Entity\EntityInterface|null
   */
  public function getEntityFromGid(ConverterInterface $converter, BaseInterface $object, Mapping $mapping, int $i): ?EntityInterface {
    foreach ($converter->getGid() as $elem) {
      if (str_starts_with($elem, 'const:')) {
        $array[$elem] = explode(':', $elem)[1];
      }
      else {
        switch ($elem) {
          case 'profile':
            $array[$elem] = $converter->getProfile()->id();
            break;
          case 'converter':
            $array[$elem] = $converter->id();
            break;
          case 'id':
            $array[$elem] = $this->getObjectValue($converter, $object, $mapping, $i);
            break;
          default:
            // We need to retrieve the MaPS object based on the value defined
            // in the GID.
            foreach ($mapping->getConverters() as $targetConverter) {
              // First we load the converter.
              if ($targetConverter !== NULL) {
                /** @var \Drupal\a12s_maps_sync\MapsApi $mapsApi */
                $mapsApi = \Drupal::service('a12s_maps_sync.maps_api');

                $filters = $targetConverter->getFiltersArray();
                $filters['id'] = [
                  'filtering_type' => 'value',
                  'value' => $this->getObjectValue($converter, $object, $mapping, $i),
                ];
                $objects = $mapsApi->getObjects($targetConverter->getProfile(), $filters);

                if (!empty($objects)) {
                  $datum = reset($objects);
                  $targetObject = MapsObject::createFromApiData($targetConverter, $datum);

                  $newMapping = clone($mapping);
                  $newMapping->setSource($elem);
                  $array[$elem] = $this->getObjectValue($converter, $targetObject, $newMapping, $i);
                  break;
                }
              }
            }
            break;
        }
      }
    }

    // Try to load the target entity.
    if ($gid = $this->getGidFromArray($converter->getGid(), $array)) {
      $entityType = !empty($mapping->getEntityType()) ? $mapping->getEntityType() : $converter->getConverterEntityType();

      $storage = \Drupal::service('entity_type.manager')
        ->getStorage($entityType);
      $entities = $storage->loadByProperties([BaseInterface::GID_FIELD => $gid]);

      if (!empty($entities)) {
        return reset($entities);
      }
    }

    return NULL;
  }

}
