<?php

namespace Drupal\a12s_maps_sync\Entity;

use Drupal\Core\Config\Entity\ConfigEntityBase;
use Drupal\Core\Entity\EntityStorageInterface;

/**
 * Defines the MaPS Sync profile entity.
 *
 * @ConfigEntityType(
 *   id = "maps_sync_profile",
 *   label = @Translation("MaPS Sync Profile"),
 *   handlers = {
 *     "view_builder" = "Drupal\Core\Entity\EntityViewBuilder",
 *     "list_builder" = "Drupal\a12s_maps_sync\ProfileListBuilder",
 *     "form" = {
 *       "add" = "Drupal\a12s_maps_sync\Form\ProfileForm",
 *       "edit" = "Drupal\a12s_maps_sync\Form\ProfileForm",
 *       "delete" = "Drupal\a12s_maps_sync\Form\ProfileDeleteForm",
 *       "import" = "Drupal\a12s_maps_sync\Form\ProfileImportForm",
 *       "attribute_import_wizard" = "Drupal\a12s_maps_sync\Form\ProfileAttributeImportWizardForm",
 *     },
 *     "route_provider" = {
 *       "html" = "Drupal\a12s_maps_sync\Routing\ProfileHtmlRouteProvider",
 *     },
 *   },
 *   config_prefix = "maps_sync_profile",
 *   admin_permission = "administer site configuration",
 *   entity_keys = {
 *     "id" = "id",
 *     "label" = "label",
 *     "uuid" = "uuid"
 *   },
 *   config_export = {
 *     "uuid",
 *     "langcode",
 *     "status",
 *     "dependencies",
 *     "id",
 *     "label",
 *     "default_maps_language",
 *     "medias_path",
 *     "python_profile_id"
 *   },
 *   links = {
 *     "canonical" = "/admin/a12s_maps_sync/profile/{maps_sync_profile}/edit",
 *     "add-form" = "/admin/a12s_maps_sync/profile/add",
 *     "edit-form" = "/admin/a12s_maps_sync/profile/{maps_sync_profile}/edit",
 *     "delete-form" = "/admin/a12s_maps_sync/profile/{maps_sync_profile}/delete",
 *     "import-form" = "/admin/a12s_maps_sync/profile/{maps_sync_profile}/import",
 *     "attribute-import-wizard-form" = "/admin/a12s_maps_sync/profile/{maps_sync_profile}/attribute-import-wizard",
 *     "collection" = "/admin/a12s_maps_sync/profile",
 *   }
 * )
 */
class Profile extends ConfigEntityBase implements ProfileInterface {

  /**
   * The MaPS Sync profile ID.
   *
   * @var string
   */
  protected string $id;

  /**
   * The MaPS Sync profile label.
   *
   * @var string
   */
  protected string $label;

  /**
   * The default MaPS System language id.
   */
  protected ?int $default_maps_language = NULL;

  /**
   * The MaPS medias path.
   */
  protected ?string $medias_path = NULL;

  /**
   * The Python profile id.
   */
  protected ?string $python_profile_id = NULL;

  /**
   * {@inheritdoc}
   */
  public function getConverters(): array {
    $query = \Drupal::service('entity_type.manager')->getStorage('maps_sync_converter')->getQuery();
    $query
      ->condition('profile_id', $this->id())
      ->sort('weight', 'ASC');

    $ids = $query->execute();
    $converters = Converter::loadMultiple($ids);

    // Order the converters by group, then by weight.
    $i = 0;
    $return = [];
    foreach (array_keys($this->getAvailableParents()) as $parent) {
      foreach ($converters as $converter) {
        if ($converter->getParent() === $parent) {
          $return[$i + $converter->getWeight()] = $converter;
        }
      }

      // Increase index by 100 so the weight don't overlap.
      $i += 100;
    }

    // Reset the converter ids as the keys.
    $converters = [];
    foreach ($return as $converter) {
      $converters[$converter->id()] = $converter;
    }

    return $converters;
  }

  /**
   * {@inheritdoc}
   */
  public function getDefaultMapslanguage(): ?int {
    return $this->default_maps_language;
  }

  /**
   * {@inheritdoc}
   */
  public function setDefaultMapslanguage(int $language): ProfileInterface {
    $this->default_maps_language = $language;
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getMediasPath(): ?string {
    return $this->medias_path;
  }

  /**
   * {@inheritdoc}
   */
  public function setMediasPath(string $medias_path): ProfileInterface {
    $this->medias_path = $medias_path;
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getPythonProfileId(): string {
    return $this->python_profile_id ?: $this->id();
  }

  /**
   * {@inheritdoc}
   */
  public function setPythonProfileId(string $python_profile_id): ProfileInterface {
    $this->python_profile_id = $python_profile_id;
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getLockName(): string {
    return 'a12s_maps_sync:lock:profile:' . $this->id();
  }

  /**
   * {@inheritdoc}
   */
  public function getAvailableParents(): array {
    $parents = [
      'medias' => t('Medias'),
      'references' => t('References'),
    ];

    $query = \Drupal::service('entity_type.manager')->getStorage('maps_sync_converter')->getQuery();
    $query->condition('profile_id', $this->id());
    $ids = $query->execute();
    $converters = Converter::loadMultiple($ids);

    foreach ($converters as $converter) {
      $parent = $converter->getParent();
      if (!isset($parents[$parent])) {
        $title = $parent;
        if (str_starts_with($parent, 'auto-config')) {
          $exploded = explode('--', $title);
          $title = 'Auto-config: ' . end($exploded);
        }

        $parents[$parent] = $title;
      }

    }

    $parents['default'] = t('Default');

    return $parents;
  }

}
