<?php

namespace Drupal\a12s_maps_sync\Form;

use Drupal\a12s_maps_sync\AutoConfigManager;
use Drupal\a12s_maps_sync\Entity\Converter;
use Drupal\a12s_maps_sync\Entity\ConverterInterface;
use Drupal\a12s_maps_sync\Entity\ProfileInterface;
use Drupal\a12s_maps_sync\MapsApi;
use Drupal\Core\Batch\BatchBuilder;
use Drupal\Core\Entity\EntityForm;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

class ProfileAttributeImportWizardForm extends EntityForm {

  /**
   * @param MapsApi $mapsApi
   * @param AutoConfigManager $autoConfigManager
   */
  public function __construct(
    protected MapsApi $mapsApi,
    protected AutoConfigManager $autoConfigManager,
  ) {}

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('a12s_maps_sync.maps_api'),
      $container->get('a12s_maps_sync.auto_config_manager'),
    );
  }

  /**
   * The current step.
   *
   * @var int
   */
  protected int $step = 1;

  /**
   * @var ConverterInterface[]
   */
  protected array $converters = [];

  /**
   * @var array
   */
  protected array $attributeSet = [];

  /**
   * @var array
   */
  protected array $attributes = [];

  /**
   * {@inheritdoc}
   */
  public function form(array $form, FormStateInterface $form_state) {
    /** @var ProfileInterface $profile */
    $profile = $this->entity;

    $mapsLanguage = $profile->getDefaultMapslanguage();

    switch ($this->step) {
      case 1:
        $converters = $profile->getConverters();
        $options = [];
        foreach ($converters as $converter) {
          $options[$converter->id()] = $converter->label();
        }

        $form['converters'] = [
          '#title' => $this->t('Select the converters'),
          '#type' => 'checkboxes',
          '#required' => TRUE,
          '#options' => $options,
        ];

        break;

      case 2:
        // Get the attribute sets.
        $attributeSets = $this->mapsApi->getConfiguration($profile, [
          'type' => 'attribute_set',
          'id_language' => $mapsLanguage,
        ]);
        $options = [];

        foreach ($attributeSets as $attributeSet) {
          $options[$attributeSet['code']] = "{$attributeSet['value']} ({$attributeSet['code']})";
        }

        asort($options);

        $form['attribute_set'] = [
          '#title' => $this->t('Select an attribute set'),
          '#description' => $this->t('Allows to filter the available attributes.'),
          '#type' => 'select',
          '#options' => $options,
          '#empty_option' => $this->t('- All attribute sets -'),
        ];

        break;

      case 3:
        // Get all available attributes.
        $attributes = $this->mapsApi->getConfiguration($profile, [
          'type' => 'attribute',
          'id_language' => $mapsLanguage,
        ]);

        $allowedAttributes = [];

        // Filter on the attribute set if needed.
        if (!empty($this->attributeSet)) {
          $attributeSetsHasAttributes = $this->mapsApi->getConfiguration($profile, ['type' => 'attribute_set_has_attribute', 'id' => $this->attributeSet['id']]);
          $attributeSetHasAttributes = reset($attributeSetsHasAttributes);

          foreach ($attributeSetHasAttributes['data'] as $attributeId) {
            $allowedAttributes[] = $attributeId;
          }
        }

        // Remove already mapped attributes.
        $mappedAttributes = [];
        foreach ($this->converters as $converter) {
          foreach ($converter->getMapping() as $mappingItem) {
            $mappedAttributes[] = $mappingItem->getSource();
          }
        }

        $options = [];
        foreach ($attributes as $attribute) {
          if (empty($allowedAttributes) || in_array($attribute['id'], $allowedAttributes)) {
            if (!in_array($attribute['code'], $mappedAttributes)) {
              $options[$attribute['id']] = "{$attribute['value']} ({$attribute['code']})";
            }
          }
        }

        asort($options);

        $form['attributes'] = [
          '#type' => 'checkboxes',
          '#title' => $this->t('Select the attributes to convert'),
          '#options' => $options,
        ];

        break;
    }

    return $form;
  }

  /**
   * Go to the next step.
   *
   * @param array $form
   * @param FormStateInterface $form_state
   * @return void
   */
  protected function nextStep(array $form, FormStateInterface $form_state) {
    $form_state->setRebuild();
    $this->step++;
  }

  /**
   * Submit the first step.
   *
   * @param array $form
   * @param FormStateInterface $form_state
   * @return void
   */
  public function submitStep1(array $form, FormStateInterface $form_state) {
    $converters = [];
    $converterIds = array_filter($form_state->getValue('converters'));

    foreach ($converterIds as $converterId) {
      $converters[] = Converter::load($converterId);
    }

    $this->converters = $converters;

    $this->nextStep($form, $form_state);
  }

  /**
   * Submit the second step.
   *
   * @param array $form
   * @param FormStateInterface $form_state
   * @return void
   */
  public function submitStep2(array $form, FormStateInterface $form_state) {
    if ($attributeSetCode = $form_state->getValue('attribute_set')) {
      $attributeSets = $this->mapsApi->getConfiguration($this->entity, ['type' => 'attribute_set', 'code' => $attributeSetCode]);

      $this->attributeSet = reset($attributeSets);
    }

    $this->nextStep($form, $form_state);
  }

  /**
   * Go to the previous step.
   *
   * @param array $form
   * @param FormStateInterface $form_state
   * @return void
   */
  public function previousStep(array $form, FormStateInterface $form_state) {
    $form_state->setRebuild();
    $this->step = min(1, $this->step - 1);
  }

  /**
   * Generate the configuration.
   *
   * @param array $form
   * @param FormStateInterface $form_state
   * @return void
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   * @throws \Drupal\Core\Entity\EntityStorageException
   */
  public function generateConfig(array $form, FormStateInterface $form_state) {
    $batch = (new BatchBuilder())
      ->setTitle($this->t('Generating configuration'))
      ->setFinishCallback([self::class, 'batchFinished']);

    foreach ($this->converters as $converter) {
      $batch->addOperation(
        [self::class, 'batchGenerate'],
        [
          $converter,
          $this->attributeSet['code'] ?? NULL,
          $form_state->getValue('attributes'),
        ]
      );
    }

    batch_set($batch->toArray());
  }

  /**
   * Batch callback.
   *
   * @param Converter $converter
   * @param $attributeSetCode
   * @param $attributeIds
   * @return void
   */
  public static function batchGenerate(Converter $converter, ?string $attributeSetCode, array $attributeIds) {
    /** @var AutoConfigManager $autoConfigManager */
    $autoConfigManager = \Drupal::service('a12s_maps_sync.auto_config_manager');

    $autoConfigManager->manageAttributeSets(
      $converter,
      [$attributeSetCode],
      AutoConfigManager::LIBRARIES_MANAGEMENT_ENTITY_REFERENCE,
      $attributeIds,
    );
  }

  /**
   * Batch finished callback.
   *
   * @return void
   */
  public static function batchFinished() {
    \Drupal::messenger()->addMessage("Configuration generated");
  }

  /**
   * {@inheritdoc}
   */
  protected function actions(array $form, FormStateInterface $form_state) {
    $actions = [];

    if ($this->step > 1) {
      $actions['previous'] = [
        '#type' => 'submit',
        '#value' => $this->t('Previous'),
        '#submit' => ['::submitForm', '::previousStep'],
      ];
    }

    if ($this->step < 3) {
      $actions['next'] = [
        '#type' => 'submit',
        '#value' => $this->t('Next'),
        '#submit' => ['::submitForm', '::submitStep' . $this->step],
      ];
    }
    else {
      $actions['generate'] = [
        '#type' => 'submit',
        '#value' => $this->t('Generate configuration'),
        '#submit' => ['::submitForm', '::generateConfig'],
      ];
    }

    return $actions;
  }

}
