<?php

namespace Drupal\a12s_maps_sync\Routing;

use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Entity\Routing\AdminHtmlRouteProvider;
use Symfony\Component\Routing\Route;

/**
 * Provides routes for Maps sync converter entities.
 *
 * @see Drupal\Core\Entity\Routing\AdminHtmlRouteProvider
 * @see Drupal\Core\Entity\Routing\DefaultHtmlRouteProvider
 */
class ConverterHtmlRouteProvider extends AdminHtmlRouteProvider {

  public function getRoutes(EntityTypeInterface $entity_type) {
    $collection = parent::getRoutes($entity_type);

    if ($filters_form_route = $this->getFiltersFormRoute($entity_type)) {
      $collection->add("entity.maps_sync_converter.filters_form", $filters_form_route);
    }

    if ($mapping_form_route = $this->getMappingFormRoute($entity_type)) {
      $collection->add("entity.maps_sync_converter.mapping_form", $mapping_form_route);
    }

    if ($import_form_route = $this->getImportFormRoute($entity_type)) {
      $collection->add("entity.maps_sync_converter.import_form", $import_form_route);
    }

    if ($auto_config_form_route = $this->getAutoConfigFormRoute($entity_type)) {
      $collection->add("entity.maps_sync_converter.auto_config_form", $auto_config_form_route);
    }

    return $collection;
  }

  /**
   * Gets the filters-form route.
   *
   * @param \Drupal\Core\Entity\EntityTypeInterface $entity_type
   *   The entity type.
   *
   * @return \Symfony\Component\Routing\Route|null
   *   The generated route, if available.
   */
  protected function getFiltersFormRoute(EntityTypeInterface $entity_type) {
    if ($entity_type->hasLinkTemplate('filters-form')) {
      $route = new Route($entity_type->getLinkTemplate('filters-form'));
      $route
        ->setDefaults([
          '_entity_form' => "maps_sync_converter.filters",
          '_title_callback' => 'Drupal\a12s_maps_sync\Form\ConverterFiltersForm::title',
        ])
        ->setRequirement('_entity_access', "maps_sync_converter.update")
        ->setOption('parameters', [
          'maps_sync_converter' => ['type' => 'entity:maps_sync_converter'],
        ]);

      // Entity types with serial IDs can specify this in their route
      // requirements, improving the matching process.
      if ($this->getEntityTypeIdKeyType($entity_type) === 'integer') {
        $route->setRequirement('maps_sync_converter', '\d+');
      }

      $this->addProfileParameterToRoute($route);
      return $route;
    }
  }

  /**
   * Gets the mapping-form route.
   *
   * @param \Drupal\Core\Entity\EntityTypeInterface $entity_type
   *   The entity type.
   *
   * @return \Symfony\Component\Routing\Route|null
   *   The generated route, if available.
   */
  protected function getMappingFormRoute(EntityTypeInterface $entity_type) {
    if ($entity_type->hasLinkTemplate('mapping-form')) {
      $route = new Route($entity_type->getLinkTemplate('mapping-form'));
      $route
        ->setDefaults([
          '_entity_form' => "maps_sync_converter.mapping",
          '_title_callback' => 'Drupal\a12s_maps_sync\Form\ConverterMappingForm::title',
        ])
        ->setRequirement('_entity_access', "maps_sync_converter.update")
        ->setOption('parameters', [
          'maps_sync_converter' => ['type' => 'entity:maps_sync_converter'],
        ]);

      // Entity types with serial IDs can specify this in their route
      // requirements, improving the matching process.
      if ($this->getEntityTypeIdKeyType($entity_type) === 'integer') {
        $route->setRequirement('maps_sync_converter', '\d+');
      }

      $this->addProfileParameterToRoute($route);
      return $route;
    }
  }

  /**
   * Gets the import-form route.
   *
   * @param \Drupal\Core\Entity\EntityTypeInterface $entity_type
   *   The entity type.
   *
   * @return \Symfony\Component\Routing\Route|null
   *   The generated route, if available.
   */
  protected function getImportFormRoute(EntityTypeInterface $entity_type) {
    if ($entity_type->hasLinkTemplate('import-form')) {
      $route = new Route($entity_type->getLinkTemplate('import-form'));
      $route
        ->setDefaults([
          '_entity_form' => "maps_sync_converter.import",
          '_title_callback' => 'Drupal\a12s_maps_sync\Form\ConverterImportForm::title',
        ])
        ->setRequirement('_permission', "import a12s maps sync converter")
        ->setOption('parameters', [
          'maps_sync_converter' => ['type' => 'entity:maps_sync_converter'],
        ]);

      // Entity types with serial IDs can specify this in their route
      // requirements, improving the matching process.
      if ($this->getEntityTypeIdKeyType($entity_type) === 'integer') {
        $route->setRequirement('maps_sync_converter', '\d+');
      }

      $this->addProfileParameterToRoute($route);
      return $route;
    }
  }

  /**
   * Gets the auto-config-form route.
   *
   * @param \Drupal\Core\Entity\EntityTypeInterface $entity_type
   *   The entity type.
   *
   * @return \Symfony\Component\Routing\Route|null
   *   The generated route, if available.
   */
  protected function getAutoConfigFormRoute(EntityTypeInterface $entity_type) {
    if ($entity_type->hasLinkTemplate('auto-config-form')) {
      $route = new Route($entity_type->getLinkTemplate('auto-config-form'));
      $route
        ->setDefaults([
          '_entity_form' => "maps_sync_converter.auto_config",
          '_title_callback' => 'Drupal\a12s_maps_sync\Form\ConverterAutoConfigForm::title',
        ])
        ->setRequirement('_entity_access', "maps_sync_converter.update")
        ->setOption('parameters', [
          'maps_sync_converter' => ['type' => 'entity:maps_sync_converter'],
        ]);

      // Entity types with serial IDs can specify this in their route
      // requirements, improving the matching process.
      if ($this->getEntityTypeIdKeyType($entity_type) === 'integer') {
        $route->setRequirement('maps_sync_converter', '\d+');
      }

      $this->addProfileParameterToRoute($route);
      return $route;
    }
  }

  /**
   * Add profile required information for the route converter.
   *
   * @param \Symfony\Component\Routing\Route $route
   */
  protected function addProfileParameterToRoute(Route $route): void {
    $parameters = $route->getOption('parameters');
    $parameters['maps_sync_profile']['type'] = 'entity:maps_sync_profile';
    $route->setOption('parameters', $parameters);
  }

  /**
   * {@inheritdoc}
   */
  protected function getAddPageRoute(EntityTypeInterface $entity_type) {
    if ($route = parent::getAddPageRoute($entity_type)) {
      $this->addProfileParameterToRoute($route);
      return $route;
    }
  }

  /**
   * {@inheritdoc}
   */
  protected function getAddFormRoute(EntityTypeInterface $entity_type) {
    if ($route = parent::getAddFormRoute($entity_type)) {
      $this->addProfileParameterToRoute($route);
      return $route;
    }
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditFormRoute(EntityTypeInterface $entity_type) {
    if ($route = parent::getEditFormRoute($entity_type)) {
      $this->addProfileParameterToRoute($route);
      return $route;
    }
  }

  /**
   * {@inheritdoc}
   */
  protected function getDeleteFormRoute(EntityTypeInterface $entity_type) {
    if ($route = parent::getDeleteFormRoute($entity_type)) {
      $this->addProfileParameterToRoute($route);
      return $route;
    }
  }

  /**
   * {@inheritdoc}
   */
  protected function getCanonicalRoute(EntityTypeInterface $entity_type) {
    if ($route = parent::getEditFormRoute($entity_type)) {
      $this->addProfileParameterToRoute($route);
      return $route;
    }
  }

  /**
   * {@inheritdoc}
   */
  protected function getCollectionRoute(EntityTypeInterface $entity_type) {
    if ($route = parent::getCollectionRoute($entity_type)) {
      $this->addProfileParameterToRoute($route);
      return $route;
    }
  }

}
