<?php

namespace Drupal\a12s_maps_sync\Maps;

use Drupal\a12s_maps_sync\Entity\ConverterInterface;
use Drupal\a12s_maps_sync\Entity\Profile;
use Drupal\a12s_maps_sync\Plugin\SourceHandler\MapsObject as MapsObjectSource;

/**
 * Class MapsObject
 */
class MapsObject extends Base implements ObjectInterface {

  /**
   * The parent object ID.
   *
   * @var int
   */
  protected int $parent_id;

  /**
   * The source object ID.
   *
   * @var int
   */
  protected int $source_id;

  /**
   * The object code.
   *
   * @var string
   */
  protected string $code;

  /**
   * The object nature.
   *
   * @var string
   */
  protected string $nature;

  /**
   * The object type.
   *
   * @var string
   */
  protected string $type;

  /**
   * The object classes.
   *
   * @var array
   */
  protected array $classes;

  /**
   * The object status.
   *
   * @var string
   */
  protected string $status;

  /**
   * The object sequence.
   *
   * @var int
   */
  protected int $sequence;

  /**
   * The object configuration type.
   *
   * @var string
   */
  protected ?string $config_type;

  /**
   * The object medias.
   *
   * @var \Drupal\a12s_maps_sync\Maps\Media[]
   */
  protected array $medias = [];

  /**
   * @var \Drupal\a12s_maps_sync\Maps\Link[]
   */
  protected array $links = [];

  /**
   * @var array
   */
  protected array $properties = [];

  /**
   * @var string|null
   */
  protected ?string $converter;

  /**
   * @param int $id
   * @param int $parent_id
   * @param int $source_id
   * @param string $code
   * @param string $nature
   * @param string $type
   * @param array $classes
   * @param string $status
   * @param int $sequence
   * @param int $imported
   * @param string|null $config_type
   * @param array $medias
   * @param array $links
   */
  public function __construct(
    int $id,
    int $parent_id,
    int $source_id,
    string $code,
    string $nature,
    string $type,
    array $classes,
    string $status,
    int $sequence,
    int $imported,
    ?string $config_type = NULL,
    array $medias = [],
    array $links = [],
    array $properties = [],
  ) {
    parent::__construct();
    $this->id = $id;
    $this->parent_id = $parent_id;
    $this->source_id = $source_id;
    $this->code = $code;
    $this->nature = $nature;
    $this->type = $type;
    $this->classes = $classes;
    $this->status = $status;
    $this->sequence = $sequence;
    $this->config_type = $config_type;
    $this->medias = $medias;
    $this->links = $links;
    $this->imported = $imported;
    $this->properties = $properties;
  }

  /**
   * {@inheritDoc}
   */
  public function get(string $name, int $language_id = NULL, $delta = NULL) {
    $properties = ['id', 'profile', 'converter', 'parent_id', 'source_id', 'code', 'nature', 'type', 'status', 'sequence', 'config_type', 'medias'];

    if (in_array($name, $properties)) {
      return $this->{$name};
    }

    if ($language_id === NULL) {
      $profile = Profile::load($this->getProfile());
      $language_id = $profile->getDefaultMapslanguage();
    }

    return parent::get($name, $language_id, $delta);
  }

  /**
   * {@inheritDoc}
   */
  public function getId(): int {
    return $this->id;
  }

  /**
   * {@inheritDoc}
   */
  public function getProfile(): string {
    return $this->profile;
  }

  /**
   * {@inheritDoc}
   */
  public function getParentId(): int {
    return $this->parent_id;
  }

  /**
   * {@inheritDoc}
   */
  public function getSourceId(): int {
    return $this->source_id;
  }

  /**
   * {@inheritDoc}
   */
  public function getCode(): string {
    return $this->code;
  }

  /**
   * {@inheritDoc}
   */
  public function getNature(): string {
    return $this->nature;
  }

  /**
   * {@inheritDoc}
   */
  public function getType(): string {
    return $this->type;
  }

  /**
   * {@inheritdoc}
   */
  public function getClasses(): array {
    return $this->classes;
  }

  /**
   * {@inheritDoc}
   */
  public function getStatus(): string {
    return $this->status;
  }

  /**
   * {@inheritDoc}
   */
  public function getSequence(): int {
    return $this->sequence;
  }

  /**
   * {@inheritDoc}
   */
  public function getConfigType(): string {
    return $this->config_type;
  }

  /**
   * {@inheritDoc}
   */
  public function getMedias(): array {
    return $this->medias;
  }

  /**
   * {@inheritDoc}
   */
  public function setMedias(array $medias): ObjectInterface {
    $this->medias = $medias;
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getLinks(?int $linkTypeId, ?bool $useQuantityAsSequence): array {
    $links = $this->links;

    if (!is_null($linkTypeId)) {
      foreach ($links as $k => $link) {
        if ($link->getTypeId() != $linkTypeId) {
          unset($links[$k]);
        }
      }
    }

    // Order by sequence.
    if ($useQuantityAsSequence) {
      $return = [];
      $toAdd = [];

      foreach ($links as $link) {
        $seq = $link->getCount();
        if (!isset($return[$seq])) {
          $return[$seq] = $link;
        }
        else {
          // Duplicate sequence, we will add it to the links array after.
          $toAdd[] = $link;
        }
      }

      $links = $return;
      $links += $toAdd;

      ksort($links);
    }

    return array_values($links);
  }

  /**
   * {@inheritdoc}
   */
  public function setLinks(array $links): ObjectInterface {
    $this->links = $links;
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function setProperties(array $properties): ObjectInterface {
    $this->properties = $properties;
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getProperties(): array {
    return $this->properties;
  }

  /**
   * {@inheritDoc}
   */
  public function getCountValues(string $name): int {
    $properties = ['id', 'profile', 'parent_id', 'source_id', 'code', 'nature', 'type', 'status', 'sequence', 'config_type'];
    if (in_array($name, $properties)) {
      return 1;
    }

    if (in_array($name, ['medias', 'links'])) {
      return count($this->{$name});
    }

    return parent::getCountValues($name);
  }

  /**
   * {@inheritDoc}
   */
  public static function createFromApiData(ConverterInterface $converter, array $data): ObjectInterface {
    $object = new MapsObject(
      $data['id'],
      $data['parent_id'],
      $data['source_id'],
      $data['code'],
      $data['nature'],
      $data['type'],
      $data['classes'],
      $data['status'],
      $data['sequence'],
      (int) $data['imported'],
      $data['config_type'],
      $data['medias'],
    );

    $object->setProfile($converter->getProfile()->id());
    $object->setConverter($converter->id());

    // Add the attributes.
    $attributes = MapsObjectSource::generateAttributes($converter->getProfile(), $data);
    $object->setAttributes($attributes);

    // Add the medias.
    $medias = [];
    foreach ($data['medias'] as $mediaData) {
      $medias[] = new \Drupal\a12s_maps_sync\Maps\Media(
        $mediaData['id'],
        $mediaData['extension'],
        $mediaData['url'],
        $mediaData['filename'],
        $mediaData['type'],
        $mediaData['weight'],
        $mediaData['imported'],
      );
    }
    $object->setMedias($medias);

    // Add the links.
    $links = [];
    foreach ($data['links'] as $linkData) {
      $links[] = new Link(
        $linkData['source_id'],
        $linkData['target_id'],
        $linkData['type_id'],
        $linkData['count'],
      );
    }
    $object->setLinks($links);

    $object->setProperties($data['properties']);

    return $object;
  }

}
