<?php

namespace Drupal\a12s_maps_sync\Converter;

use Drupal\a12s_maps_sync\Entity\Converter;
use Drupal\a12s_maps_sync\Entity\ConverterInterface;

class Mapping {

  /**
   * Status for manually created mapping.
   */
  public const MAPPING_STATUS_MANUAL = 0;

  /**
   * Status for automatically created mapping.
   */
  public const MAPPING_STATUS_AUTO = 1;

  /**
   * Status for automatically created but override mapping.
   */
  public const MAPPING_STATUS_OVERRIDE = 2;

  /**
   * Mapping required status.
   */
  public const MAPPING_REQUIRED_ERROR = 1;

  /**
   * Mapping required status.
   */
  public const MAPPING_REQUIRED_SKIP = 2;

  /**
   * @param string $source
   * @param string $target
   * @param string $handler
   * @param bool $append
   * @param int $status
   * @param array $options
   */
  public function __construct(
    protected string $source,
    protected string $target,
    protected string $handler,
    protected bool $append = FALSE,
    protected int $status = self::MAPPING_STATUS_MANUAL,
    protected bool $required = FALSE,
    protected ?int $requiredBehavior = self::MAPPING_REQUIRED_ERROR,
    protected array $options = [],
  ) {}

  /**
   * @return string
   */
  public function getSource(): string {
    return $this->source;
  }

  /**
   * @param string $source
   *
   * @return Mapping
   */
  public function setSource(string $source): Mapping {
    $this->source = $source;
    return $this;
  }

  /**
   * @return string
   */
  public function getTarget(): string {
    return $this->target;
  }

  /**
   * @param string $target
   *
   * @return Mapping
   */
  public function setTarget(string $target): Mapping {
    $this->target = $target;
    return $this;
  }


  /**
   * @return bool|null
   */
  public function isAppend(): ?bool {
    return $this->append;
  }

  /**
   * @param bool|null $append
   *
   * @return Mapping
   */
  public function setAppend(?bool $append): Mapping {
    $this->append = $append;
    return $this;
  }

  /**
   * @return string
   */
  public function getHandlerId(): string {
    return $this->handler;
  }

  /**
   * @param string $handler
   *
   * @return Mapping
   */
  public function setHandler(string $handler): Mapping {
    $this->handler = $handler;
    return $this;
  }

  /**
   * @return int
   */
  public function getStatus(): int {
    return $this->status ?? 0;
  }

  /**
   * @param int $status
   *
   * @return Mapping
   */
  public function setStatus(int $status): Mapping {
    $this->status = $status;
    return $this;
  }

  /**
   * @return array
   */
  public function getOptions(): array {
    return $this->options;
  }

  /**
   * @param array $options
   *
   * @return $this
   */
  public function setOptions(array $options): Mapping {
    $this->options = $options;
    return $this;
  }

  /**
   * @param string $option
   *
   * @return mixed|null
   */
  public function getOption(string $option): mixed {
    return !empty($this->options[$option]) ? $this->options[$option] : NULL;
  }

  /**
   * @param string $option
   * @param $value
   *
   * @return $this
   */
  public function setOption(string $option, $value): Mapping {
    $this->options[$option] = $value;
    return $this;
  }

  /**
   * @return \Drupal\a12s_maps_sync\Entity\ConverterInterface[]
   */
  public function getConverters(): array {
    $converters = [];

    if (!empty($this->getOption('converters'))) {
      foreach ($this->getOption('converters') as $converterId) {
        $converters[] = Converter::load($converterId);
      }
    }

    return $converters;
  }

  /**
   * @param \Drupal\a12s_maps_sync\Entity\ConverterInterface[] $converter
   *
   * @return Mapping
   */
  public function setConverters(array $converters): Mapping {
    $ids = [];

    /** @var ConverterInterface $converter */
    foreach ($converters as $converter) {
      $ids[] = $converter->id();
    }

    $this->setOption('converters', $ids);
    return $this;
  }

  /**
   * @return string|null
   */
  public function getEntityType(): ?string {
    return $this->getOption('entity_type');
  }

  /**
   * @param string|null $entityType
   *
   * @return Mapping
   */
  public function setEntityType(?string $entityType): Mapping {
    $this->setOption('entity_type', $entityType);
    return $this;
  }


  /**
   * @return string|null
   */
  public function getContextualizedAttributeType(): ?string {
    return $this->getOption('contextualized_attribute_type');
  }

  /**
   * @param string|null $contextualizedAttributeType
   *
   * @return Mapping
   */
  public function setContextualizedAttributeType(?string $contextualizedAttributeType): Mapping {
    $this->setOption('contextualized_attribute_type', $contextualizedAttributeType);
    return $this;
  }

  /**
   * @return string|null
   */
  public function getContextualizedAttributeValueField(): ?string {
    return $this->getOption('contextualized_attribute_value_field');
  }

  /**
   * @param string|null $contextualizedAttributeValueField
   *
   * @return Mapping
   */
  public function setContextualizedAttributeValueField(?string $contextualizedAttributeValueField): Mapping {
    $this->setOption('contextualized_attribute_value_field', $contextualizedAttributeValueField);
    return $this;
  }

  /**
   * @return string|null
   */
  public function getContextualizedAttributeCriteriaField(): ?string {
    return $this->getOption('contextualized_attribute_criteria_field');
  }

  /**
   * @param string|null $contextualizedAttributeCriteriaField
   *
   * @return Mapping
   */
  public function setContextualizedAttributeCriteriaField(?string $contextualizedAttributeCriteriaField): Mapping {
    $this->setOption('contextualized_attribute_criteria_field', $contextualizedAttributeCriteriaField);
    return $this;
  }

  /**
   * @return bool
   */
  public function isRequired(): bool {
    return (bool) $this->required;
  }

  /**
   * @param bool $required
   *
   * @return $this
   */
  public function setRequired(bool $required): Mapping {
    $this->required = $required;
    return $this;
  }

  /**
   * @return int|null
   */
  public function getRequiredBehavior(): ?int {
    return $this->requiredBehavior;
  }

  /**
   * @param int|null $requiredBehavior
   *
   * @return $this
   */
  public function setRequiredBehavior(?int $requiredBehavior): Mapping {
    $this->requiredBehavior = $requiredBehavior;
    return $this;
  }

  /**
   * @return array
   */
  public function toArray(): array {
    return [
      'source' => $this->source,
      'target' => $this->target,
      'handler' => $this->handler,
      'append' => $this->append,
      'required' => $this->required,
      'requiredBehavior' => $this->requiredBehavior,
      'status' => $this->status,
      'options' => $this->options,
    ];
  }

}
