<?php

declare(strict_types=1);

namespace Drupal\a12s_maps_sync\Form;

use Drupal\a12s_maps_sync\Entity\Converter;
use Drupal\a12s_maps_sync\Entity\ConverterInterface;
use Drupal\a12s_maps_sync\Entity\ProfileInterface;
use Drupal\a12s_maps_sync\MapsApi;
use Drupal\a12s_maps_sync\Plugin\SourceHandlerPluginManager;
use Drupal\Core\Entity\EntityForm;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Class ConverterForm.
 */
class ConverterForm extends EntityForm {

  use ApiDependantFormTrait;

  /**
   * @param \Drupal\a12s_maps_sync\MapsApi $mapsApi
   *   The MaPS API service.
   * @param \Drupal\a12s_maps_sync\Plugin\SourceHandlerPluginManager $sourceHandlerPluginManager
   */
  public function __construct(
    protected MapsApi $mapsApi,
    protected SourceHandlerPluginManager $sourceHandlerPluginManager,
  ) {}

  /**
   * {@inheritdoc}
   * @noinspection PhpParamsInspection
   */
  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('a12s_maps_sync.maps_api'),
      $container->get('plugin.manager.maps_sync_source_handler'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function form(array $form, FormStateInterface $form_state, ProfileInterface $profile = NULL): array {
    $form = parent::form($form, $form_state);

    if ($profile === NULL) {
      $profile = \Drupal::request()->get('maps_sync_profile');
    }

    $form['#profile'] = $profile;

    /** @var \Drupal\a12s_maps_sync\Entity\ConverterInterface $converter */
    $converter = $this->entity;
    $form['label'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Label'),
      '#maxlength' => 255,
      '#default_value' => $converter->label(),
      '#description' => $this->t('Label for the Maps sync converter.'),
      '#required' => TRUE,
    ];

    $form['id'] = [
      '#type' => 'machine_name',
      '#default_value' => $converter->id(),
      '#machine_name' => [
        'exists' => '\Drupal\a12s_maps_sync\Entity\Converter::load',
      ],
      '#disabled' => !$converter->isNew(),
    ];

    $form['profile_id'] = [
      '#type' => 'value',
      '#value' => $profile->id(),
    ];

    $plugins = \Drupal::service('plugin.manager.maps_sync_handler');

    $form['handler_id'] = [
      '#type' => 'select',
      '#title' => $this->t('Handler'),
      '#options' => array_map(fn($plugin_definition) => $plugin_definition['label'], $plugins->getDefinitions()),
      '#default_value' => $converter->getHandlerId() ?: 'default',
      '#required' => TRUE,
    ];

    $form['gid'] = [
      '#type' => 'textfield',
      '#title' => $this->t('GID'),
      '#description' => $this->t('GIDs, separated by commas'),
      '#default_value' => $converter->getGid() ? implode(',', $converter->getGid()) : 'profile,converter,id',
      '#required' => TRUE,
      '#multiple' => TRUE,
    ];

    $form['parent'] = [
      '#type' => 'select',
      '#title' => $this->t('Parent group'),
      '#options' => $profile->getAvailableParents(),
      '#default_value' => $converter->getParent(),
      '#required' => TRUE,
    ];

    $form['source'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Source'),
    ];

    $form['source']['maps_type'] = [
      '#type' => 'select',
      '#title' => $this->t('MaPS type'),
      '#default_value' => $converter->getMapsType(),
      '#required' => TRUE,
      '#options' => $this->getMapsTypeOptions(),
    ];

    $form['target'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Target'),
    ];

    $entity_types = array_filter(\Drupal::config('a12s_maps_sync.settings')->get('allowed_entity_types'));

    $options = [];
    foreach ($entity_types as $entity_type_id) {
      $entity_type = \Drupal::entityTypeManager()->getDefinition($entity_type_id);
      $options[$entity_type->id()] = $entity_type->getLabel();
    }
    asort($options);

    $form['target']['entity_type'] = [
      '#type' => 'select',
      '#options' => $options,
      '#title' => $this->t('Entity type'),
      '#default_value' => $converter->getConverterEntityType(),
      '#required' => TRUE,
    ];

    foreach ($entity_types as $entity_type_id) {
      $bundles = \Drupal::service('entity_type.bundle.info')->getBundleInfo($entity_type_id);

      if (!empty($bundles)) {
        $options = array_map(fn($bundle) => $bundle['label'], $bundles);
        asort($options);

        $form['target']['bundle_' . $entity_type_id] = [
          '#type' => 'select',
          '#options' => $options,
          '#title' => $this->t('Bundle'),
          '#default_value' => $converter->getConverterBundle(),
          '#states' => [
            'visible' => [
              ':input[name="entity_type"]' => ['value' => $entity_type_id],
            ],
          ],
        ];
      }
    }

    $form['status'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Status'),
    ];

    $form['status']['status_property'] = [
      '#type' => 'select',
      '#title' => $this->t('Status property'),
//      '#options' => $this->getStatusPropertyOptions($profile),
      '#options' => [
        'status' => $this->t('Status'),
        'properties' => $this->t('Property'),
        'attributes' => $this->t('Attribute'),
      ],
      '#empty_options' => $this->t('- Select -'),
      '#description' => $this->t("Defines the property used for the status"),
      '#default_value' => $converter->getStatusProperty() ?? 'status',
    ];

    $userInput = $form_state->getUserInput();

    $form['status']['status_property_name'] = [
      '#type' => 'select',
      '#title' => $this->t('Status property name'),
      '#options' => $this->getStatusPropertyOptions($profile, $converter->getMapsType() ?? ($userInput['maps_type'] ?? 'object')),
      '#empty_options' => $this->t('- Select -'),
      '#description' => $this->t("Defines the property used for the status"),
      '#default_value' => $converter->getStatusPropertyName(),
      '#states' => [
        'visible' => [
          'select[name="status_property"]' => [
            ['value' => 'attributes'],
            ['value' => 'properties']
          ],
        ],
      ],
    ];

    $form['status']['published_statuses'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Published statuses'),
      '#description' => $this->t('Published statuses codes, separated by commas'),
      '#default_value' => $converter->getPublishedStatuses() ? implode(',', $converter->getPublishedStatuses()) : '',
      '#states' => [
        'visible' => [
          'select[name="maps_type"]' => ['value' => 'object'],
        ],
      ],
    ];

    $form['status']['unpublished_statuses'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Unpublished statuses'),
      '#description' => $this->t('Unpublished statuses codes, separated by commas'),
      '#default_value' => $converter->getUnpublishedStatuses() ? implode(',', $converter->getUnpublishedStatuses()) : '',
      '#states' => [
        'visible' => [
          'select[name="maps_type"]' => ['value' => 'object'],
        ],
      ],
    ];

    $form['status']['deleted_statuses'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Deleted statuses'),
      '#description' => $this->t('Deleted statuses codes, separated by commas'),
      '#default_value' => $converter->getDeletedStatuses() ? implode(',', $converter->getDeletedStatuses()) : '',
      '#states' => [
        'visible' => [
          'select[name="maps_type"]' => ['value' => 'object'],
        ],
      ],
    ];

    $form['status']['status_management'] = [
      '#type' => 'select',
      '#title' => $this->t('Status management'),
      '#description' => $this->t('Defines how the unpublished statuses are managed.'),
      '#default_value' => $converter->getStatusManagement() ?? '',
      '#options' => [
        ConverterInterface::STATUS_MANAGEMENT_UNPUBLISH => $this->t('Unpublish the entity (if possible)'),
        ConverterInterface::STATUS_MANAGEMENT_DELETE => $this->t('Delete the entity'),
        ConverterInterface::STATUS_MANAGEMENT_UPDATE_EXISTING_IGNORE_STATUS => $this->t('Update the entity but keep the status unchanged'),
      ],
      '#empty_option' => $this->t('- Select -'),
    ];

    $form['status']['media_status_published_value'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Status published value'),
      '#description' => $this->t("Defines the value used for the media’s published status"),
      '#states' => [
        'visible' => [
          'select[name="status_management"]' => ['value' => ConverterInterface::STATUS_MANAGEMENT_DELETE],
          'select[name="maps_type"]' => ['value' => 'media'],
        ],
      ],
      '#default_value' => $converter->getMediaStatusPublishedValue() ?? '',
    ];

    return $form;
  }

  /**
   * @param \Drupal\Core\Entity\EntityInterface $entity
   * @param array $form
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   */
  protected function copyFormValuesToEntity(EntityInterface $entity, array $form, FormStateInterface $form_state) {
    $values = $form_state->getValues();

    // Only keep the needed bundle (because of Form states).
    $bundle_key = 'bundle_' . $form_state->getValue('entity_type');
    if ($bundle = $form_state->getValue($bundle_key)) {
      $values['bundle'] = $bundle;
    }

    $values['published_statuses'] = explode(',', $values['published_statuses']);
    $values['unpublished_statuses'] = explode(',', $values['unpublished_statuses']);
    $values['deleted_statuses'] = explode(',', $values['deleted_statuses']);
    $values['gid'] = explode(',', $values['gid']);

    foreach ($values as $key => $value) {
      // Remove useless bundles.
      if (!str_starts_with($key, 'bundle_')) {
        $entity->set($key, $value);
      }
    }
  }

  /**
   * {@inheritdoc}
   * @throws \Drupal\Core\Entity\EntityStorageException
   * @throws \Drupal\Core\Entity\EntityMalformedException
   */
  public function save(array $form, FormStateInterface $form_state): void {
    $maps_sync_converter = $this->entity;
    $status = $maps_sync_converter->save();

    switch ($status) {
      case SAVED_NEW:
        $this->messenger()->addMessage($this->t('Created the %label Maps sync converter.', [
          '%label' => $maps_sync_converter->label(),
        ]));
        break;

      default:
        $this->messenger()->addMessage($this->t('Saved the %label Maps sync converter.', [
          '%label' => $maps_sync_converter->label(),
        ]));
    }
    $form_state->setRedirectUrl($maps_sync_converter->toUrl('collection'));
  }

  /**
   * @return array
   */
  protected function getMapsTypeOptions(): array {
    $options = [];
    foreach ($this->sourceHandlerPluginManager->getDefinitions() as $definition) {
      $options[$definition['id']] = $definition['label'];
    }

    return $options;
  }

  /**
   * Get the status property options based on the given profile and type.
   *
   * @param ProfileInterface $profile
   *   The profile instance.
   * @param string|null $type
   *   (Optional) The type of source handler to use. Defaults to 'object'.
   *
   * @return array
   *   An array of filter types.
   */
  public function getStatusPropertyOptions(ProfileInterface $profile, ?string $type = 'object'): array {
    // Create fake converter.
    $converter = new Converter([], 'maps_sync_converter');
    $converter->setProfileId($profile->id());

    $sourceHandler = $this->sourceHandlerPluginManager->createInstance($type);
    return $sourceHandler->getFilterTypes($converter);
  }

}
