<?php

namespace Drupal\a12s_maps_sync\Plugin\MappingHandler;

use Drupal\a12s_maps_sync\Converter\Mapping;
use Drupal\a12s_maps_sync\Entity\ContextualizedAttribute;
use Drupal\a12s_maps_sync\Entity\ConverterInterface;
use Drupal\a12s_maps_sync\Maps\BaseInterface;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Language\LanguageInterface;

/**
 * Plugin implementation of the a12s_maps_sync_mapping_handler.
 *
 * @MappingHandler(
 *   id = "criteria_attribute",
 *   label = @Translation("Criteria Attribute"),
 *   description = @Translation("Criteria Attribute.")
 * )
 */
class CriteriaAttributeHandler extends EntityReferenceHandler {

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
      'source_converter' => NULL,
      'converters' => NULL,
      'entity_type' => 'contextualized_attribute',
      'contextualized_attribute_type' => NULL,
      'contextualized_attribute_value_field' => NULL,
      'contextualized_attribute_criteria_field' => NULL,
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form = parent::buildConfigurationForm($form, $form_state);

    $bundles = \Drupal::service('entity_type.bundle.info')->getBundleInfo('contextualized_attribute');
    $options = [];
    if (!empty($bundles)) {
      foreach ($bundles as $bundle_id => $bundle) {
        $options[$bundle_id] = $bundle['label'];
      }
      asort($options);
    }
    $form['contextualized_attribute_type'] = [
      '#type' => 'select',
      '#title' => $this->t('Contextualized attribute type'),
      '#empty_option' => $this->t('- Select -'),
      '#options' => $options,
      '#default_value' => $this->getConfiguration()['contextualized_attribute_type'],
      '#description' => $this->t('The contextualized attribute type, with a criteria field and a value field.'),
    ];

    // contextualized_attribute fields.
    /** @var \Drupal\Core\Entity\EntityFieldManagerInterface $entityFieldManager */
    $entityFieldManager = \Drupal::service('entity_field.manager');
    // @todo Use ajax to only show field instances for the selected bundle.
    $fields = $entityFieldManager->getFieldStorageDefinitions('contextualized_attribute');

    $options = [];
    foreach ($fields as $id => $field) {
      $options[$id] = $id;
    }

    asort($options);

    $form['contextualized_attribute_criteria_field'] = [
      '#type' => 'select',
      '#title' => $this->t('Criteria field'),
      '#empty_option' => $this->t('- Select -'),
      '#options' => $options,
      '#default_value' => $this->getConfiguration()['contextualized_attribute_criteria_field'],
      '#description' => $this->t('The field in the contextualized attribute entity, that holds the criteria entity.'),
    ];

    $form['contextualized_attribute_value_field'] = [
      '#type' => 'select',
      '#title' => $this->t('Value field'),
      '#empty_option' => $this->t('- Select -'),
      '#options' => $options,
      '#default_value' => $this->getConfiguration()['contextualized_attribute_value_field'],
      '#description' => $this->t('The field in the contextualized attribute entity, that holds the MaPS attribute value.'),
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function mapData(EntityInterface $entity, BaseInterface $object, Mapping $mapping, int $mapsLanguage, ?LanguageInterface $language = NULL) {
    // Only process this once.
    if (!$language || $language->getId() !== \Drupal::languageManager()->getDefaultLanguage()->getId()) {
      return NULL;
    }

    $source = $mapping->getSource();
    $targetFieldName = $mapping->getTarget();

    // Start by removing current paragraphs.
    foreach ($entity->{$targetFieldName} as $fieldValue) {
      if ($fieldValue->entity) {
        $contextualizedAttribute = ContextualizedAttribute::load($fieldValue->entity->id());
        if ($contextualizedAttribute) {
          $contextualizedAttribute->delete();
        }
      }
    }
    $entity->{$targetFieldName} = [];

    // @todo refactor the following with the code from the parent...

    $value = NULL;
    foreach ($mapping->getConverters() as $targetConverter) {
      // We need to find the entity in Drupal.
      $entityType = !empty($mapping->getEntityType()) ? $mapping->getEntityType() : $targetConverter->getConverterEntityType();

      /** @var \Drupal\Core\Entity\EntityFieldManager $entityFieldManager */
      $entityFieldManager = \Drupal::service('entity_field.manager');
      $fields = $entityFieldManager->getFieldStorageDefinitions($entity->getEntityType()
        ->id());

      // Ensure that we have a field name configured in the mapping.
      if (empty($targetFieldName)) {
        \Drupal::logger('a12s_maps_sync')
          ->error('No target field name defined for target converter %converter', ['%converter' => $targetConverter->id()]);
        return FALSE;
      }

      if (empty($fields[$targetFieldName])) {
        \Drupal::logger('a12s_maps_sync')
          ->error('The field %field_name does not exist.', ['%field_name' => $targetFieldName]);
        return FALSE;
      }

      if ($targetFieldName === 'parent') {
        $is_multiple = FALSE;
      }
      else {
        $is_multiple = $fields[$targetFieldName]->get('cardinality') !== 1;
      }

      // @todo manage this in a clean way, this is too "hacky".
      $target_gid = $targetConverter->getGid();

      // Delete the previous value.
      if ($mapping->isAppend()) {
        $entity->{$mapping->getTarget()} = [];
      }

      for ($i = 0; $i < $object->getCountValues($source); $i++) {
        $targetEntity = $this->getEntityFromGid($targetConverter, $object, $mapping, $i);

        if (!is_null($targetEntity )) {
          // Create the new ContextualizedAttribute.
          $_value = $object->get($source, NULL, $i);

          $contextualizedAttribute = ContextualizedAttribute::create([
            'bundle' => $mapping->getContextualizedAttributeType(),
            $mapping->getContextualizedAttributeCriteriaField() => ['target_id' => $targetEntity->id()],
            $mapping->getContextualizedAttributeValueField() => $_value,
          ]);
          $contextualizedAttribute->save();

          // Set the field value.
          $value = $this->getValue($contextualizedAttribute);

          if ($is_multiple) {
            $entity->{$targetFieldName}[] = $value;
          }
          else {
            $entity->{$targetFieldName} = $value;
          }
        }
    }
    }

    return $value;
  }

  /**
   * {@inheritdoc}
   */
  public function getObjectValue(ConverterInterface $converter, BaseInterface $object, Mapping $mapping, int $i) {
    return $object->getAttribute($mapping->getSource())->getData()[$i]['idobject_criteria'];
  }

}
