/**
 * @file
 * Converter behaviors.
 */
(function ($, window, Drupal, once) {

  /**
   * Move a converter in the converters table between regions via select list.
   *
   * This behavior is dependent on the tableDrag behavior, since it uses the
   * objects initialized in that behavior to update the row.
   *
   * @type {Drupal~behavior}
   *
   * @prop {Drupal~behaviorAttach} attach
   *   Attaches the tableDrag behavior for converters in converter administration.
   */
  Drupal.behaviors.converterDrag = {
    attach(context, settings) {
      // tableDrag is required and we should be on the converters admin page.
      if (
        typeof Drupal.tableDrag === 'undefined' ||
        typeof Drupal.tableDrag.converters === 'undefined'
      ) {
        return;
      }

      /**
       * Function to check empty regions and toggle classes based on this.
       *
       * @param {jQuery} table
       *   The jQuery object representing the table to inspect.
       * @param {jQuery} rowObject
       *   The jQuery object representing the table row.
       */
      function checkEmptyregions(table, rowObject) {
        table.find('tr.region-message').each(function () {
          const $this = $(this);
          // If the dragged row is in this region, but above the message row,
          // swap it down one space.
          if ($this.prev('tr').get(0) === rowObject.element) {
            // Prevent a recursion problem when using the keyboard to move rows
            // up.
            if (
              rowObject.method !== 'keyboard' ||
              rowObject.direction === 'down'
            ) {
              rowObject.swap('after', this);
            }
          }
          // This region has become empty.
          if (
            $this.next('tr').is(':not(.draggable)') ||
            $this.next('tr').length === 0
          ) {
            $this.removeClass('region-populated').addClass('region-empty');
          }
          // This region has become populated.
          else if ($this.is('.region-empty')) {
            $this.removeClass('region-empty').addClass('region-populated');
          }
        });
      }

      /**
       * Function to update the last placed row with the correct classes.
       *
       * @param {jQuery} table
       *   The jQuery object representing the table to inspect.
       * @param {jQuery} rowObject
       *   The jQuery object representing the table row.
       */
      function updateLastPlaced(table, rowObject) {
        // Remove the color-success class from new converter if applicable.
        table.find('.color-success').removeClass('color-success');

        const $rowObject = $(rowObject);
        if (!$rowObject.is('.drag-previous')) {
          table.find('.drag-previous').removeClass('drag-previous');
          $rowObject.addClass('drag-previous');
        }
      }

      /**
       * Update converter weights in the given region.
       *
       * @param {jQuery} table
       *   Table with draggable items.
       * @param {string} region
       *   Machine name of region containing converters to update.
       */
      function updateConverterWeights(table, region) {
        // Calculate minimum weight.
        let weight = -Math.round(table.find('.draggable').length / 2);
        // Update the converter weights.
        table
          .find(`.region-${region}-message`)
          .nextUntil('.region-title')
          .find('select.converter-weight')
          .each(function () {
            // Increment the weight before assigning it to prevent using the
            // absolute minimum available weight. This way we always have an
            // unused upper and lower bound, which makes manually setting the
            // weights easier for users who prefer to do it that way.
            this.value = ++weight;
          });
      }

      const table = $('#converters');
      // Get the converters tableDrag object.
      const tableDrag = Drupal.tableDrag.converters;
      // Add a handler for when a row is swapped, update empty regions.
      tableDrag.row.prototype.onSwap = function (swappedRow) {
        checkEmptyregions(table, this);
        updateLastPlaced(table, this);
      };

      // Add a handler so when a row is dropped, update fields dropped into
      // new regions.
      tableDrag.onDrop = function () {
        const dragObject = this;
        const $rowElement = $(dragObject.rowObject.element);
        // Use "region-message" row instead of "region" row because
        // "region-{region_name}-message" is less prone to regexp match errors.
        const regionRow = $rowElement.prevAll('tr.region-message').get(0);
        const regionName = regionRow.className.replace(
          /([^ ]+[ ]+)*region-([^ ]+)-message([ ]+[^ ]+)*/,
          '$2',
        );
        const regionField = $rowElement.find('select.converter-region-select');
        // Check whether the newly picked region is available for this converter.
        if (regionField.find(`option[value=${regionName}]`).length === 0) {
          // If not, alert the user and keep the converter in its old region
          // setting.
          window.alert(Drupal.t('The converter cannot be placed in this region.'));
          // Simulate that there was a selected element change, so the row is
          // put back to from where the user tried to drag it.
          regionField.trigger('change');
        }

        // Update region and weight fields if the region has been changed.
        if (!regionField.is(`.converter-region-${regionName}`)) {
          const weightField = $rowElement.find('select.converter-weight');
          const oldregionName = weightField[0].className.replace(
            /([^ ]+[ ]+)*converter-weight-([^ ]+)([ ]+[^ ]+)*/,
            '$2',
          );
          regionField
            .removeClass(`converter-region-${oldregionName}`)
            .addClass(`converter-region-${regionName}`);
          weightField
            .removeClass(`converter-weight-${oldregionName}`)
            .addClass(`converter-weight-${regionName}`);
          regionField[0].value = regionName;
        }

        updateConverterWeights(table, regionName);
      };

      // Add the behavior to each region select list.
      $(once('converter-region-select', 'select.converter-region-select', context)).on(
        'change',
        function (event) {
          // Make our new row and select field.
          const row = $(this).closest('tr');
          const select = $(this);
          // Find the correct region and insert the row as the last in the
          // region.
          tableDrag.rowObject = new tableDrag.row(row[0]);
          const regionMessage = table.find(
            `.region-${select[0].value}-message`,
          );
          const regionItems = regionMessage.nextUntil(
            '.region-message, .region-title',
          );
          if (regionItems.length) {
            regionItems.last().after(row);
          }
          // We found that regionMessage is the last row.
          else {
            regionMessage.after(row);
          }
          updateConverterWeights(table, select[0].value);
          // Modify empty regions with added or removed fields.
          checkEmptyregions(table, tableDrag.rowObject);
          // Update last placed converter indication.
          updateLastPlaced(table, row);
          // Show unsaved changes warning.
          if (!tableDrag.changed) {
            $(Drupal.theme('tableDragChangedWarning'))
              .insertBefore(tableDrag.table)
              .hide()
              .fadeIn('slow');
            tableDrag.changed = true;
          }
          // Remove focus from selectbox.
          select.trigger('blur');
        },
      );
    },
  };
})(jQuery, window, Drupal, once);
