<?php

namespace Drupal\a12s_maps_sync;

use Drupal\a12s_maps_sync\Entity\Converter;
use Drupal\a12s_maps_sync\Entity\ConverterInterface;
use Drupal\a12s_maps_sync\Entity\Profile;
use Drupal\a12s_maps_sync\Entity\ProfileInterface;
use Drupal\a12s_maps_sync\Exception\MapsException;
use Drupal\a12s_maps_sync\Maps\BaseInterface;
use Drupal\Core\Entity\EntityInterface;

class BatchService {

  /**
   * @param \Drupal\a12s_maps_sync\Entity\ProfileInterface $profile
   * @param int $limit
   *
   * @return array
   */
  public static function getProfileImportBatchDefinition(ProfileInterface $profile, int $limit, bool $releaseLock = TRUE): array {
    $operations = [];

    foreach ($profile->getConverters() as $converter) {
      $subDefinition = self::getConverterImportBatchDefinition($converter, $limit, FALSE);
      $operations = array_merge($operations, $subDefinition['operations']);
    }

    if ($releaseLock) {
      $operations[] = [static::class . '::releaseLock', [$profile->getLockName()]];
    }

    return [
      'title' => t('Processing profile @profile import', ['@profile' => $profile->label()]),
      'operations' => $operations,
      'finished' => static::class . '::processConverterImportFinished',
    ];
  }

  /**
   * @param \Drupal\a12s_maps_sync\Entity\ConverterInterface $converter
   * @param int $limit
   * @param bool $releaseLock
   *
   * @return array
   */
  public static function getConverterImportBatchDefinition(ConverterInterface $converter, int $limit, bool $releaseLock = TRUE): array {
    $fromTime = $converter->getLastImportedTime();

    // Get the total of elements.
    $count = $converter->getSourceHandler()->getCountData($converter, $converter->getFiltersArray(), $fromTime);
    $total = $limit > 0 ? (int) ceil($count / $limit) : 1;

    $batch = [
      'title' => t('Processing converter @converter import', ['@converter' => $converter->label()]),
      'operations' => [],
      'finished' => static::class . '::processConverterImportFinished',
    ];

    for ($i = 0; $i < $total; $i++) {
      $batch['operations'][] = [static::class . '::processConverterImport', [$converter, $limit, $i, $total]];
    }

    if ($releaseLock) {
      $batch['operations'][] = [static::class . '::releaseLock', [$converter->getLockName()]];
    }

    return $batch;
  }

  /**
   * @param string $mapsType
   * @param int $id
   * @param \Drupal\a12s_maps_sync\Entity\ConverterInterface|null $converter
   *
   * @param bool $withDependencies
   *
   * @return array
   */
  public static function getObjectImportBatchDefinition(int $id, ?ConverterInterface $converter, bool $withDependencies = FALSE): array {
    $objects = $converter->getSourceHandler()->getData($converter, ['id' => ['value' => $id]], 1);

    if (!empty($objects)) {
      $object = reset($objects);
      $operations = self::getObjectImportOperations($withDependencies, $converter, $object);

      return [
        'title' => t('Processing import of @maps_type @id', [
          '@maps_type' => 'object',
          '@id' => $id,
        ]),
        'operations' => $operations,
        'finished' => static::class . '::genericFinishedCallback',
      ];
    }

    return [];
  }

  /**
   * @param \Drupal\Core\Entity\EntityInterface $entity
   * @param bool $withDependencies
   * @param \Drupal\a12s_maps_sync\Entity\ProfileInterface|null $profile
   * @param \Drupal\a12s_maps_sync\Entity\ConverterInterface|null $converter
   *
   * @return array
   * @throws \Drupal\a12s_maps_sync\Exception\MapsException
   */
  public static function getEntityImportBatchDefinition(EntityInterface $entity, bool $withDependencies = FALSE, ?ProfileInterface $profile = NULL, ?ConverterInterface $converter = NULL): array {
    // Ensure that the entity has a gid.
    if (!$entity->hasField(BaseInterface::GID_FIELD) || $entity->get(BaseInterface::GID_FIELD)->isEmpty()) {
      throw new MapsException("The entity {$entity->getEntityTypeId()} {$entity->id()} has no GID");
    }

    if (is_null($converter)) {
      $profiles = !is_null($profile) ? [$profile] : Profile::loadMultiple();
      foreach ($profiles as $profile) {
        // Get the converters.
        foreach ($profile->getConverters() as $_converter) {
          if ($_converter->getConverterEntityType() === $entity->getEntityTypeId() && $_converter->getConverterBundle() === $entity->bundle()) {
            $converter = $_converter;
            break 2;
          }
        }
      }
    }

    if (is_null($converter)) {
      throw new MapsException("No matching converter for {$entity->getEntityTypeId()} {$entity->bundle()}");
    }

    // Ensure that the converter's gid allowed us to force the re-import.
    $gid = $converter->getGid();
    foreach ($gid as $part) {
      if (in_array($part, ['id', 'source_id', 'code'])) {
        $gidKey = $part;
        break;
      }
    }

    if (!isset($gidKey)) {
      throw new MapsException("Unable to retrieve the MaPS object with the converter GID");
    }

    $object = $converter->getSourceHandler()->findObjectFromEntity($converter, $entity);
    if (!$object) {
      throw new MapsException("Unable to retrieve the MaPS object");
    }

    $operations = self::getObjectImportOperations($withDependencies, $converter, $object);
    return [
      'title' => t('Processing import of entity @entity_type @entity_id', [
        '@entity_type' => $entity->getEntityType()->getLabel(),
        '@entity_id' => $entity->id(),
      ]),
      'operations' => $operations,
      'finished' => static::class . '::genericFinishedCallback',
    ];
  }

  /**
   * Release the given lock.
   *
   * @param string $lockName
   * @param array $context
   */
  public static function releaseLock(string $lockName, array &$context): void {
    $context['message'] = t('Purge lock');
    \Drupal::state()->delete($lockName);
  }

  /**
   * Process the import of the converter.
   *
   * @param \Drupal\a12s_maps_sync\Entity\Converter $converter
   * @param int $limit
   * @param int $step
   * @param int $total
   * @param array $context
   */
  public static function processConverterImport(Converter $converter, int $limit, int $step, int $total, array &$context): void {
    $context['message'] = t('Importing converter @converter (@step / @total)',
      [
        '@converter' => $converter->label(),
        '@step' => $step + 1,
        '@total' => $total,
      ]
    );

    $results = $converter->import($limit);

    $context['converter'] = $converter->id();
    $context['results'][] = $results;
  }

  /**
   * @param \Drupal\a12s_maps_sync\Entity\Converter $converter
   * @param string $objectId
   * @param array $context
   *
   * @return void
   */
  public static function processObjectIdImport(Converter $converter, string $objectId, array &$context): void {
    $gid = $converter->getGid();
    $sourceHandler = $converter->getSourceHandler();

    foreach ($gid as $part) {
      if (in_array($part, $sourceHandler->getAllowedGidEntityKeys())) {
        $key = $part;
        break;
      }
    }

    $data = $sourceHandler->getData($converter, [$key => $objectId], 1);
    if (!empty($data)) {
      $object = reset($data);
      self::processObjectImport($converter, $object, $context);
    }
  }

  /**
   * Process the import of an object.
   *
   * @param \Drupal\a12s_maps_sync\Entity\Converter $converter
   * @param \Drupal\a12s_maps_sync\Maps\BaseInterface $object
   * @param $context
   */
  public static function processObjectImport(Converter $converter, BaseInterface $object, &$context) {
    $context['message'] = t('Importing object @object with converter @converter',
      [
        '@converter' => $converter->label(),
        '@object' => $object->getId(),
        '@step' => 1,
        '@total' => 1,
      ]
    );

    $results = $converter->importObject($object);

    $context['converter'] = $converter->id();
    $context['results'][] = $results;
  }

  /**
   * Batch Finished callback.
   *
   * @param bool $success
   *   Success of the operation.
   * @param array $results
   *   Array of results for post-processing.
   * @param array $operations
   *   Array of operations.
   */
  public static function processConverterImportFinished($success, array $results, array $operations): void {
    $messenger = \Drupal::messenger();

    if ($success) {
      if (!empty($results)) {
        $count = 0;
        foreach ($results as $result) {
          $count += count($result);
        }
        $messenger->addMessage(t('Import finished: @count results processed.', ['@count' => $count]));
      }
      else {
        $messenger->addMessage(t('Nothing to import.'));
      }
    }
    else {
      // An error occurred.
      // $operations contains the operations that remained unprocessed.
      $error_operation = reset($operations);
      $messenger->addError(
        t('An error occurred while processing @operation with arguments : @args',
          [
            '@operation' => $error_operation[0],
            '@args' => print_r($error_operation[0], TRUE),
          ]
        )
      );
    }
  }

  /**
   * @param bool $withDependencies
   * @param \Drupal\a12s_maps_sync\Entity\ConverterInterface|null $converter
   * @param mixed $object
   * @param $entity
   *
   * @return array
   */
  protected static function getObjectImportOperations(bool $withDependencies, ?ConverterInterface $converter, mixed $object): array {
    $operations = [];

    if ($withDependencies) {
      $dependencies = $converter->getObjectDependencies($object);
      foreach ($dependencies as $dependency) {
        $depConverter = $dependency['converter'];

        foreach ($dependency['values'] as $value) {
          // For links, we have to do the same process for each link's target...
          if ($dependency['with_dependencies']) {
            $depObjects = $converter->getSourceHandler()->getData($converter, ['id' => $value], 1);
            if (!empty($depObjects)) {
              $depObject = reset($depObjects);
              $operations = array_merge($operations, self::getObjectImportOperations(TRUE, $depConverter, $depObject));
            }
          }
          else {
            $operations[] = [
              static::class . '::processObjectIdImport',
              [$depConverter, $value]
            ];
          }
        }
      }
    }

    $operations[] = [
      static::class . '::processObjectImport',
      [$converter, $object]
    ];

    return $operations;
  }

  /**
   * @return void
   */
  public static function genericFinishedCallback(): void {
    \Drupal::messenger()->addMessage(t('Import finished'));
  }

}
