<?php

namespace Drupal\a12s_maps_sync;

use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Manipulates entity type information.
 *
 * This class contains primarily bridged hooks for compile-time or
 * cache-clear-time hooks. Runtime hooks should be placed in EntityOperations.
 *
 * Taken form devel module...
 */
class EntityTypeInfo implements ContainerInjectionInterface {

  use StringTranslationTrait;

  /**
   * The current user.
   *
   * @var \Drupal\Core\Session\AccountInterface
   */
  protected AccountInterface $currentUser;

  /**
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * EntityTypeInfo constructor.
   *
   * @param \Drupal\Core\Session\AccountInterface $current_user
   *   Current user.
   */
  public function __construct(AccountInterface $current_user, EntityTypeManagerInterface $entityTypeManager) {
    $this->currentUser = $current_user;
    $this->entityTypeManager = $entityTypeManager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('current_user'),
      $container->get('entity_type.manager'),
    );
  }

  /**
   * Adds import entity links to appropriate entity types.
   *
   * This is an alter hook bridge.
   *
   * @param \Drupal\Core\Entity\EntityTypeInterface[] $entity_types
   *   The master entity type list to alter.
   *
   * @see hook_entity_type_alter()
   */
  public function entityTypeAlter(array &$entity_types) {
    $allowedEntityTypes = \Drupal::config('a12s_maps_sync.settings')->get('allowed_entity_types');

    if (!empty($allowedEntityTypes)) {
      $entityTypes = array_filter($allowedEntityTypes);

      foreach ($entity_types as $entityTypeId => $entityType) {
        if (in_array($entityTypeId, $entityTypes)) {
          if (($entityType->getFormClass('default') || $entityType->getFormClass('edit')) && $entityType->hasLinkTemplate('edit-form')) {
            // We use edit-form template to extract and set additional parameters
            // dynamically.
            $entityLink = $entityType->getLinkTemplate('edit-form');
            $this->setEntityTypeLinkTemplate($entityType, $entityLink, 'reimport-entity', "/a12s-maps-sync/reimport-entity/$entityTypeId");
          }
        }
      }
    }
  }

  /**
   * @param \Drupal\Core\Entity\EntityTypeInterface $entity_type
   * @param $entity_link
   * @param $a12s_link_key
   * @param $base_path
   *
   * @return void
   */
  protected function setEntityTypeLinkTemplate(EntityTypeInterface $entity_type, $entity_link, $a12s_link_key, $base_path) {
    $path_parts = $this->getPathParts($entity_link);
    $entity_type->setLinkTemplate($a12s_link_key, $base_path . $path_parts);
  }

  /**
   * Get path parts.
   *
   * @param string $entity_path
   *   Entity path.
   *
   * @return string
   *   Path parts.
   */
  protected function getPathParts($entity_path) {
    $path = '';
    if (preg_match_all('/{\w*}/', $entity_path, $matches)) {
      foreach ($matches[0] as $match) {
        $path .= "/$match";
      }
    }
    return $path;
  }

  /**
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *
   * @return array
   * @throws \Drupal\Core\Entity\EntityMalformedException
   */
  public function entityOperation(EntityInterface $entity) {
    $operations = [];
    if ($this->currentUser->hasPermission('reimport a12s maps sync entities')) {
      if ($entity->hasLinkTemplate('reimport-entity')) {
        $operations['reimport_entity'] = [
          'title' => $this->t('Re-import entity'),
          'weight' => 100,
          'url' => $entity->toUrl('reimport-entity'),
        ];
      }
    }
    return $operations;
  }

}
