<?php

namespace Drupal\a12s_maps_sync\Form;

use Drupal\a12s_maps_sync\Entity\ConverterInterface;
use Drupal\a12s_maps_sync\MapsApi;
use Drupal\a12s_maps_sync\Plugin\SourceHandlerPluginManager;
use Drupal\Component\Utility\NestedArray;
use Drupal\Core\Entity\EntityForm;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Class ConverterFiltersForm.
 */
class ConverterFiltersForm extends EntityForm {

  /**
   * @param \Drupal\a12s_maps_sync\MapsApi $mapsApi
   * @param \Drupal\a12s_maps_sync\Plugin\SourceHandlerPluginManager $sourceHandlerPluginManager
   */
  public function __construct(
    protected MapsApi $mapsApi,
    protected SourceHandlerPluginManager $sourceHandlerPluginManager,
  ) {}

  /**
   * {@inheritdoc}
   * @noinspection PhpParamsInspection
   */
  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('a12s_maps_sync.maps_api'),
      $container->get('plugin.manager.maps_sync_source_handler'),
    );
  }

  /**
   * Title callback.
   *
   * @return \Drupal\Core\StringTranslation\TranslatableMarkup
   */
  public function title(): TranslatableMarkup {
    return $this->t("Edit filters for converter %converter", ['%converter' => \Drupal::request()->get('maps_sync_converter')->label()]);
  }

  /**
   * {@inheritdoc}
   */
  public function form(array $form, FormStateInterface $form_state): array {
    $form = parent::form($form, $form_state);

    /** @var \Drupal\a12s_maps_sync\Entity\ConverterInterface $converter */
    $converter = $this->entity;

    // Filters management.
    $form['filters'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Filters'),
      '#prefix' => '<div id="filters-wrapper">',
      '#suffix' => '</div>',
      '#attributes' => [
        'class' => ['container-inline'],
      ],
      '#tree' => TRUE,
    ];

    $filters = $converter->getFilters();

    $countFilters = $form_state->get('count_filters');
    if ($countFilters === NULL) {
      $countFilters = max(count($filters), 1);
      $form_state->set('count_filters', $countFilters);
    }

    for ($i = 0; $i < $countFilters; $i++) {
      $filter = !empty($filters[$i]) ? $filters[$i] : NULL;

      $form['filters'][$i] = [
        '#type' => 'fieldset',
//        '#prefix' => '<div id="converter-filter-' . $i . '">',
//        '#suffix' => '</div>',
      ];

      $form['filters'][$i]['type'] = [
        '#type' => 'select',
        '#title' => $this->t('Type'),
        '#empty_option' => $this->t('- Select -'),
        '#options' => $this->getFilterTypesOptions($converter),
        '#default_value' => !is_null($filter) ? $filter->getType() : NULL,
        '#ajax' => [
          'callback' => '::updateFilterValues',
          'event' => 'change',
          'wrapper' => 'filters-wrapper'
        ],
      ];

      $type = $form_state->getValue(['filters', $i, 'type']) ?: (!empty($filters[$i]) ? $filters[$i]->getType() : NULL);
      $value = $form_state->getValue(['filters', $i, 'value']) ?: (!empty($filters[$i]) ? $filters[$i]->getValue() : NULL);
      $filteringType = $form_state->getValue(['filters', $i, 'filtering_type']) ?: (!empty($filters[$i]) ? $filters[$i]->getFilteringType() : 'value');

      if ($type) {
        $form['filters'][$i]['filtering_type'] = $this->getFilterFilteringTypeFormElement($type, $filteringType);
      }

      $triggeringElement = $form_state->getTriggeringElement();
      $parents = $triggeringElement['#parents'] ?? [];

      if (count($parents) === 3 && $parents[0] === 'filters' && end($parents) === 'type') {
        array_pop($parents);
        $parents[] = 'value';
        $userInput = $form_state->getUserInput();
        NestedArray::unsetValue($userInput, $parents);
        $form_state->setUserInput($userInput);
      }

      if ($filteringType === 'value') {
        $form['filters'][$i]['value'] = $this->getFilterValuesFormElement($converter, $type, $value);
      }
    }

    $form['filters']['actions'] = [
      '#type' => 'actions',
    ];

    $form['filters']['actions']['add_filter'] = [
      '#type' => 'submit',
      '#value' => $this->t('Add filter'),
      '#submit' => ['::addFilter'],
      '#ajax' => [
        'callback' => '::addFilterCallback',
        'wrapper' => 'filters-wrapper',
      ],
    ];

    return $form;
  }

  /**
   * @param \Drupal\Core\Entity\EntityInterface $entity
   * @param array $form
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   */
  protected function copyFormValuesToEntity(EntityInterface $entity, array $form, FormStateInterface $form_state): void {
    $values = $form_state->getValues();

    // Some post process.
    unset($values['filters']['actions']);
    $filters = [];
    foreach ($values['filters'] as $filterData) {
      if (!empty($filterData['type']) && (!empty($filterData['value'])) || $filterData['filtering_type'] !== 'value') {
        $filters[] = $filterData;
      }
    }

    $entity->set('filters', $filters);
  }

  /**
   * {@inheritdoc}
   * @throws \Drupal\Core\Entity\EntityStorageException
   * @throws \Drupal\Core\Entity\EntityMalformedException
   */
  public function save(array $form, FormStateInterface $form_state): void {
    $maps_sync_converter = $this->entity;
    $status = $maps_sync_converter->save();

    \Drupal::messenger()->addMessage($this->t('Saved filters for the the %label converter.', [
      '%label' => $maps_sync_converter->label(),
    ]));

    $form_state->setRedirectUrl($maps_sync_converter->toUrl('collection'));
  }

  /**
   * @param array $form
   * @param \Drupal\Core\Form\FormStateInterface $formState
   *
   * @return array
   */
  public function updateFilterValues(array &$form, FormStateInterface $formState): array {
    return $form['filters'];
  }

  /**
   * @param \Drupal\a12s_maps_sync\Entity\ConverterInterface $converter
   *
   * @return array
   * @throws \Drupal\Component\Plugin\Exception\PluginException
   */
  protected function getFilterTypesOptions(ConverterInterface $converter): array {
    $sourceHandler = $this->sourceHandlerPluginManager->createInstance($converter->getMapsType());
    return $sourceHandler->getFilterTypes($converter);
  }

  /**
   * @param \Drupal\a12s_maps_sync\Entity\ConverterInterface $converter
   * @param string $type
   *
   * @return array
   * @throws \Drupal\Component\Plugin\Exception\PluginException
   */
  protected function getFilterValuesOptions(ConverterInterface $converter, string $type): array {
    $sourceHandler = $this->sourceHandlerPluginManager->createInstance($converter->getMapsType());
    return $sourceHandler->getFilterValues($converter, $type);
  }

  /**
   * Submit handler for the "add-filter" button.
   *
   * Increments the max counter and causes a rebuild.
   */
  public function addFilter(array &$form, FormStateInterface $form_state): void {
    $name_field = $form_state->get('count_filters');
    $add_button = $name_field + 1;
    $form_state->set('count_filters', $add_button);
    $form_state->setRebuild();
  }

  /**
   * Callback for both ajax-enabled buttons.
   *
   * Selects and returns the fieldset with the names in it.
   */
  public function addFilterCallback(array &$form, FormStateInterface $form_state) {
    return $form['filters'];
  }

  /**
   * @param \Drupal\a12s_maps_sync\Entity\ConverterInterface $converter
   * @param string|null $filterType
   * @param string|null $filterValue
   *
   * @return array
   * @throws \Drupal\Component\Plugin\Exception\PluginException
   */
  protected function getFilterValuesFormElement(ConverterInterface $converter, ?string $filterType, ?string $filterValue): array {
    $element = [
      '#type' => 'textfield',
      '#title' => $this->t('Value'),
      '#default_value' => $filterValue,
      '#size' => 20,
    ];

    // @todo manage exceptions elsewhere, like in the source handlers?
    if (!is_null($filterType)) {
      if (!str_starts_with($filterType, 'attribute_') && $filterType !== 'parent_id' && $filterType !== 'extension') {
        $element['#type'] = 'select';
        unset($element['#size']);

        $options = $this->getFilterValuesOptions($converter, $filterType);
        asort($options);

        $element += [
          '#empty_option' => $this->t('- Select -'),
          '#options' => $options,
        ];
      }
    }
    return $element;
  }

  /**
   * Generates a form element for filtering type selection based on filter type.
   *
   * @param string $filterType
   *   The type of the filter to determine the form element presentation.
   * @param string|null $filteringType
   *   The selected filtering type or null if not defined.
   *
   * @return array
   *   A renderable array representing the form element.
   */
  protected function getFilterFilteringTypeFormElement(string $filterType, ?string $filteringType): array {
    $element = [
      '#type' => 'hidden',
      '#default_value' => $filteringType,
    ];

    // @todo manage exceptions elsewhere, like in the source handlers?
    if ($filterType && str_starts_with($filterType, 'attribute_')) {
      $element = [
        '#type' => 'select',
        '#title' => $this->t('Filtering type'),
        '#options' => [
          'value' => $this->t('Value equals'),
          'empty' => $this->t('Value is empty'),
          'exist' => $this->t('Value exists'),
          'not_exist' => $this->t('Value does not exist'),
        ],
        '#default_value' => $filteringType ?? 'value',
        '#ajax' => [
          'callback' => '::updateFilterValues',
          'event' => 'change',
          'wrapper' => 'filters-wrapper'
        ],
      ];
    }

    return $element;
  }

  /**
   * {@inheritdoc}
   */
  protected function actions(array $form, FormStateInterface $form_state): array {
    return [
      'submit' => [
        '#type' => 'submit',
        '#value' => $this->t('Save'),
        '#submit' => ['::submitForm', '::save'],
      ]
    ];
  }

}
