<?php

namespace Drupal\a12s_maps_sync\Form;

use Drupal\a12s_maps_sync\AutoConfigManager;
use Drupal\a12s_maps_sync\Entity\Converter;
use Drupal\a12s_maps_sync\Entity\ConverterInterface;
use Drupal\a12s_maps_sync\Entity\ProfileInterface;
use Drupal\a12s_maps_sync\MapsApi;
use Drupal\Core\Batch\BatchBuilder;
use Drupal\Core\Entity\EntityForm;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

class ProfileAttributeImportWizardForm extends EntityForm {

  /**
   * @param MapsApi $mapsApi
   * @param AutoConfigManager $autoConfigManager
   */
  public function __construct(
    protected MapsApi $mapsApi,
    protected AutoConfigManager $autoConfigManager,
  ) {}

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('a12s_maps_sync.maps_api'),
      $container->get('a12s_maps_sync.auto_config_manager'),
    );
  }

  /**
   * The current step.
   *
   * @var int
   */
  protected int $step = 1;

  /**
   * @var ConverterInterface[]
   */
  protected array $converters = [];

  /**
   * @var array
   */
  protected array $attributeSet = [];

  /**
   * @var array
   */
  protected array $attributes = [];

  /**
   * @var array
   */
  protected array $options = [];

  /**
   * {@inheritdoc}
   */
  public function form(array $form, FormStateInterface $form_state): array {
    /** @var ProfileInterface $profile */
    $profile = $this->entity;

    $mapsLanguage = $profile->getDefaultMapslanguage();

    switch ($this->step) {
      case 1:
        $converters = $profile->getConverters();
        $options = [];
        foreach ($converters as $converter) {
          $options[$converter->id()] = $converter->label();
        }

        $form['converters'] = [
          '#title' => $this->t('Select the converters'),
          '#type' => 'checkboxes',
          '#required' => TRUE,
          '#options' => $options,
        ];

        break;

      case 2:
        // Get the attribute sets.
        $attributeSets = $this->mapsApi->getConfiguration($profile, [
          'type' => 'attribute_set',
          'id_language' => $mapsLanguage,
        ]);
        $options = [];

        foreach ($attributeSets as $attributeSet) {
          $options[$attributeSet['code']] = "{$attributeSet['value']} ({$attributeSet['code']})";
        }

        asort($options);

        $form['attribute_set'] = [
          '#title' => $this->t('Select an attribute set'),
          '#description' => $this->t('Allows to filter the available attributes.'),
          '#type' => 'select',
          '#options' => $options,
          '#empty_option' => $this->t('- All attribute sets -'),
        ];

        break;

      case 3:
        // Get all available attributes.
        $attributes = $this->mapsApi->getConfiguration($profile, [
          'type' => 'attribute',
          'id_language' => $mapsLanguage,
        ]);

        $allowedAttributes = [];

        // Filter on the attribute set if needed.
        if (!empty($this->attributeSet)) {
          $attributeSetsHasAttributes = $this->mapsApi->getConfiguration($profile, ['type' => 'attribute_set_has_attribute', 'id' => $this->attributeSet['id']]);
          $attributeSetHasAttributes = reset($attributeSetsHasAttributes);

          foreach ($attributeSetHasAttributes['data'] as $attributeId) {
            $allowedAttributes[] = $attributeId;
          }
        }

        // Remove already mapped attributes.
        $mappedAttributes = [];
        foreach ($this->converters as $converter) {
          foreach ($converter->getMapping() as $mappingItem) {
            $mappedAttributes[] = $mappingItem->getSource();
          }
        }

        $options = [];
        foreach ($attributes as $attribute) {
          if (empty($allowedAttributes) || in_array($attribute['id'], $allowedAttributes)) {
            if (!in_array($attribute['code'], $mappedAttributes)) {
              $options[$attribute['id']] = "{$attribute['value']} ({$attribute['code']})";
            }
          }
        }

        asort($options);

        $form['attributes'] = [
          '#type' => 'checkboxes',
          '#title' => $this->t('Select the attributes to convert'),
          '#options' => $options,
        ];

        break;

      case 4:
        $form = [
          '#type' => 'container',
        ];

        $form['options']['field_prefix'] = [
          '#type' => 'textfield',
          '#title' => $this->t('Field prefix'),
          '#description' => $this->t('The field prefix (without final underscore). May be empty.'),
          '#default_value' => 'maps',
        ];

        $form['options']['vocabulary_prefix'] = [
          '#type' => 'textfield',
          '#title' => $this->t('Vocabulary prefix'),
          '#description' => $this->t('The vocabulary prefix (without final underscore). It is used for vocabulary created for managing object attributes or libraries. May be empty.'),
          '#default_value' => 'maps',
        ];

        $form['options']['manage_display'] = [
          '#type' => 'checkbox',
          '#title' => $this->t('Manage display'),
          '#description' => $this->t('Experimental: Attempt to update the entity display.')
        ];

        $form['options']['manage_form_display'] = [
          '#type' => 'checkbox',
          '#title' => $this->t('Manage form display'),
          '#description' => $this->t('Experimental: Attempt to update the entity form display.')
        ];

        $form['options']['field_group_prefix'] = [
          '#type' => 'textfield',
          '#title' => $this->t('Field group prefix'),
          '#description' => $this->t('The field group prefix (without final underscore). May be empty.'),
          '#default_value' => 'group_htab_maps',
          '#states' => [
            'visible' => [
              [':input[name="manage_display"]' => ['checked' => TRUE]],
              [':input[name="manage_form_display"]' => ['checked' => TRUE]],
            ],
          ],
        ];

        break;
    }

    return $form;
  }

  /**
   * Go to the next step.
   *
   * @param array $form
   * @param FormStateInterface $form_state
   * @return void
   */
  protected function nextStep(array $form, FormStateInterface $form_state): void {
    $form_state->setRebuild();
    $this->step++;
  }

  /**
   * Submit the first step.
   *
   * @param array $form
   * @param FormStateInterface $form_state
   * @return void
   */
  public function submitStep1(array $form, FormStateInterface $form_state): void {
    $converters = [];
    $converterIds = array_filter($form_state->getValue('converters'));

    foreach ($converterIds as $converterId) {
      $converters[] = Converter::load($converterId);
    }

    $this->converters = $converters;

    $this->nextStep($form, $form_state);
  }

  /**
   * Submit the second step.
   *
   * @param array $form
   * @param FormStateInterface $form_state
   *
   * @return void
   * @throws \Drupal\a12s_maps_sync\Exception\MapsApiException
   * @throws \GuzzleHttp\Exception\GuzzleException
   */
  public function submitStep2(array $form, FormStateInterface $form_state): void {
    if ($attributeSetCode = $form_state->getValue('attribute_set')) {
      $attributeSets = $this->mapsApi->getConfiguration($this->entity, ['type' => 'attribute_set', 'code' => $attributeSetCode]);

      $this->attributeSet = reset($attributeSets);
    }

    $this->nextStep($form, $form_state);
  }

  /**
   * @param array $form
   * @param \Drupal\Core\Form\FormStateInterface $formState
   *
   * @return void
   */
  public function submitStep3(array $form, FormStateInterface $formState): void {
    $this->attributes = $formState->getValue('attributes');
    $this->nextStep($form, $formState);
  }

  /**
   * Go to the previous step.
   *
   * @param array $form
   * @param FormStateInterface $form_state
   * @return void
   */
  public function previousStep(array $form, FormStateInterface $form_state): void {
    $form_state->setRebuild();
    $this->step = min(1, $this->step - 1);
  }

  /**
   * Generate the configuration.
   *
   * @param array $form
   * @param FormStateInterface $form_state
   * @return void
   */
  public function generateConfig(array $form, FormStateInterface $form_state): void {
    $batch = (new BatchBuilder())
      ->setTitle($this->t('Generating configuration'))
      ->setFinishCallback([self::class, 'batchFinished']);

    foreach ($this->converters as $converter) {
      $batch->addOperation(
        [self::class, 'batchGenerate'],
        [
          $converter,
          $this->attributeSet['code'] ?? NULL,
          $this->attributes,
          $form_state->getValues(),
        ]
      );
    }

    batch_set($batch->toArray());
  }

  /**
   * Batch callback.
   *
   * @param Converter $converter
   * @param string|null $attributeSetCode
   * @param array $attributeIds
   * @param array $options
   *
   * @return void
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   * @throws \Drupal\Core\Entity\EntityStorageException
   */
  public static function batchGenerate(Converter $converter, ?string $attributeSetCode, array $attributeIds, array $options = []): void {
    /** @var AutoConfigManager $autoConfigManager */
    $autoConfigManager = \Drupal::service('a12s_maps_sync.auto_config_manager');

    $autoConfigManager->manageAttributeSets(
      $converter,
      [$attributeSetCode],
      AutoConfigManager::LIBRARIES_MANAGEMENT_ENTITY_REFERENCE,
      $attributeIds,
      $options,
    );
  }

  /**
   * Batch finished callback.
   *
   * @return void
   */
  public static function batchFinished(): void {
    \Drupal::messenger()->addMessage("Configuration generated");
  }

  /**
   * {@inheritdoc}
   */
  protected function actions(array $form, FormStateInterface $form_state): array {
    $actions = [];

    if ($this->step > 1) {
      $actions['previous'] = [
        '#type' => 'submit',
        '#value' => $this->t('Previous'),
        '#submit' => ['::submitForm', '::previousStep'],
      ];
    }

    if ($this->step < 4) {
      $actions['next'] = [
        '#type' => 'submit',
        '#value' => $this->t('Next'),
        '#submit' => ['::submitForm', '::submitStep' . $this->step],
      ];
    }
    else {
      $actions['generate'] = [
        '#type' => 'submit',
        '#value' => $this->t('Generate configuration'),
        '#submit' => ['::submitForm', '::generateConfig'],
      ];
    }

    return $actions;
  }

}
