<?php

namespace Drupal\a12s_maps_sync;

use Drupal\Component\Serialization\Json;
use Drupal\Component\Utility\Html;
use Drupal\Core\Config\Entity\ConfigEntityListBuilder;
use Drupal\Core\Config\Entity\DraggableListBuilder;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Form\FormBuilderInterface;
use Drupal\Core\Form\FormInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\Theme\ThemeManagerInterface;
use Drupal\Core\Url;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\Request;

/**
 * Provides a listing of Maps sync converter entities.
 */
class ConverterListBuilder extends ConfigEntityListBuilder implements FormInterface {

  protected $weightKey = 'weight';

  /**
   * The theme containing the converters.
   *
   * @var string
   */
  protected $theme;

  /**
   * The current request.
   *
   * @var \Symfony\Component\HttpFoundation\Request
   */
  protected $request;

  /**
   * The form builder.
   *
   * @var \Drupal\Core\Form\FormBuilderInterface
   */
  protected $formBuilder;

  /**
   * The messenger.
   *
   * @var \Drupal\Core\Messenger\MessengerInterface
   */
  protected $messenger;

  /**
   * Constructs a new converterListBuilder object.
   *
   * @param \Drupal\Core\Entity\EntityTypeInterface $entity_type
   *   The entity type definition.
   * @param \Drupal\Core\Entity\EntityStorageInterface $storage
   *   The entity storage class.
   * @param \Drupal\Core\Form\FormBuilderInterface $form_builder
   *   The form builder.
   * @param \Drupal\Core\Messenger\MessengerInterface $messenger
   *   The messenger service.
   */
  public function __construct(EntityTypeInterface $entity_type, EntityStorageInterface $storage, FormBuilderInterface $form_builder, MessengerInterface $messenger) {
    parent::__construct($entity_type, $storage);

    $this->formBuilder = $form_builder;
    $this->messenger = $messenger;
    $this->limit = FALSE;
  }

  /**
   * {@inheritdoc}
   */
  public static function createInstance(ContainerInterface $container, EntityTypeInterface $entity_type) {
    return new static(
      $entity_type,
      $container->get('entity_type.manager')->getStorage($entity_type->id()),
      $container->get('form_builder'),
      $container->get('messenger')
    );
  }

  /**
   * {@inheritdoc}
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The current request.
   *
   * @return array
   *   The converter list as a renderable array.
   */
  public function render($theme = NULL, Request $request = NULL) {
    $this->request = $request;

    return $this->formBuilder->getForm($this);
  }

  /**
   * {@inheritdoc}
   */
  protected function getTitle() {
    $profile = \Drupal::request()->get('maps_sync_profile');
    return $this->t('Converters for profile @profile', ['@profile' => $profile->label()]);
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $form['#attached']['library'][] = 'core/drupal.tableheader';
    $form['#attached']['library'][] = 'a12s_maps_sync/converter';
    $form['#attributes']['class'][] = 'clearfix';

    // Build the form tree.
    $form['converters'] = $this->buildConvertersForm();

    $form['actions'] = [
      '#tree' => FALSE,
      '#type' => 'actions',
    ];
    $form['actions']['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Save converters'),
      '#button_type' => 'primary',
    ];

    return $form;
  }

  /**
   * Builds the main "Converters" portion of the form.
   *
   * @return array
   */
  protected function buildConvertersForm() {
    // Build converters first for each region.
    $converters = [];
    $entities = $this->load();
    /** @var \Drupal\a12s_maps_sync\Entity\ConverterInterface[] $entities */
    foreach ($entities as $id => $entity) {
      $converters[$entity->getParent()][$id] = [
        'label' => $entity->label(),
        'entity_id' => $id,
        'weight' => $entity->getWeight(),
        'entity' => $entity,
        'maps_type' => $entity->getMapsType(),
        'entity_type' => $entity->getConverterEntityType(),
        'bundle' => $entity->getConverterBundle(),
      ];
    }

    $form = [
      '#type' => 'table',
      '#header' => [
        $this->t('Converter'),
        $this->t('MaPS Type'),
        $this->t('Entity type'),
        $this->t('Bundle'),
        $this->t('Group'),
        $this->t('Weight'),
        $this->t('Operations'),
      ],
      '#attributes' => [
        'id' => 'converters',
      ],
    ];

    // Weights range from -delta to +delta, so delta should be at least half
    // of the amount of converters present. This makes sure all converters in the same
    // region get a unique weight.
    $weight_delta = round(count($entities) / 2);

    if (isset($entity)) {
      // Loop over each region and build converters.
      $parents = $entity->getProfile()->getAvailableParents();

      foreach ($parents as $parent => $title) {
        $form['#tabledrag'][] = [
          'action' => 'match',
          'relationship' => 'sibling',
          'group' => 'converter-region-select',
          'subgroup' => 'converter-region-' . $parent,
          'hidden' => FALSE,
        ];
        $form['#tabledrag'][] = [
          'action' => 'order',
          'relationship' => 'sibling',
          'group' => 'converter-weight',
          'subgroup' => 'converter-weight-' . $parent,
        ];

        $form['parent-' . $parent] = [
          '#attributes' => [
            'class' => ['region-title', 'region-title-' . $parent],
            'no_striping' => TRUE,
          ],
        ];
        $form['parent-' . $parent]['title'] = [
          '#markup' =>  $title,
          '#wrapper_attributes' => [
            'colspan' => 7,
          ],
        ];
        $form['parent-' . $parent . '-message'] = [
          '#attributes' => [
            'class' => [
              'region-message',
              'region-' . $parent . '-message',
              empty($converters[$parent]) ? 'region-empty' : 'region-populated',
            ],
          ],
        ];
        $form['parent-' . $parent . '-message']['message'] = [
          '#markup' => '<em>' . $this->t('No converters in this group') . '</em>',
          '#wrapper_attributes' => [
            'colspan' => 7,
          ],
        ];

        if (isset($converters[$parent])) {
          foreach ($converters[$parent] as $info) {
            /** @var \Drupal\a12s_maps_sync\Entity\ConverterInterface $converter */
            $converter = $info['entity'];
            $id = $converter->id();

            $form[$id] = [
              '#attributes' => [
                'class' => ['draggable'],
              ],
            ];

            $form[$id]['info'] = [
              '#wrapper_attributes' => [
                'class' => ['converter'],
              ],
            ];

            $form[$id]['info']['#plain_text'] = $info['label'];

            $form[$id]['maps_type'] = [
              '#markup' => $info['maps_type'],
            ];

            $form[$id]['entity_type'] = [
              '#markup' => $info['entity_type'],
            ];

            $form[$id]['bundle'] = [
              '#markup' => $info['bundle'],
            ];

            $form[$id]['parent'] = [
              '#type' => 'select',
              '#default_value' => $parent,
              '#required' => TRUE,
              '#title' => $this->t('Parent for @converter converter', ['@converter' => $info['label']]),
              '#title_display' => 'invisible',
              '#options' => $parents,
              '#attributes' => [
                'class' => ['converter-region-select', 'converter-region-' . $parent],
              ],
              '#parents' => ['converters', $id, 'region'],
            ];
            $form[$id]['weight'] = [
              '#type' => 'weight',
              '#default_value' => $info['weight'],
              '#delta' => $weight_delta,
              '#title' => $this->t('Weight for @converter converter', ['@converter' => $info['label']]),
              '#title_display' => 'invisible',
              '#attributes' => [
                'class' => ['converter-weight', 'converter-weight-' . $parent],
              ],
            ];
            $form[$id]['operations'] = $this->buildOperations($info['entity']);
          }
        }
      }
    }

    // Do not allow disabling the main system content converter when it is present.
    if (isset($form['system_main']['region'])) {
      $form['system_main']['region']['#required'] = TRUE;
    }

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'maps_sync_converter.list_builder';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEntityIds() {
    $request = \Drupal::request();
    $profile = $request->get('maps_sync_profile');

    if (!$profile) {
      throw new \InvalidArgumentException("Missing profile.");
    }

    $query = $this->getStorage()->getQuery()
      ->condition('profile_id', $profile->id())
      ->sort($this->entityType->getKey('id'));

    return $query->execute();
  }

  /**
   * {@inheritdoc}
   */
  public function getOperations(EntityInterface $entity) {
    $operations = parent::getOperations($entity);

    $operations['filters'] = [
      'title' => $this->t('Filters'),
      'url' => $entity->toUrl('filters-form'),
    ];

    $operations['mapping'] = [
      'title' => $this->t('Mapping'),
      'url' => $entity->toUrl('mapping-form'),
    ];

    $operations['import'] = [
      'title' => $this->t('Import'),
      'url' => $entity->toUrl('import-form'),
    ];

    $operations['auto-config'] = [
      'title' => $this->t('Auto config'),
      'url' => $entity->toUrl('auto-config-form'),
    ];

    return $operations;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    if (empty($form_state->getValue('converters'))) {
      $form_state->setErrorByName('converters', $this->t('No converters settings to update.'));
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $converters = $form_state->getValue('converters');
    $entities = $this->storage->loadMultiple(array_keys($converters));
    /** @var \Drupal\a12s_maps_sync\Entity\ConverterInterface[] $entities */
    foreach ($entities as $entity_id => $entity) {
      $entity_values = $form_state->getValue(['converters', $entity_id]);
      $entity->setWeight($entity_values['weight']);
      $entity->setParent($entity_values['region']);
      $entity->save();
    }
    $this->messenger->addStatus($this->t('The converters settings have been updated.'));
  }

}
