<?php

namespace Drupal\a12s_maps_sync\Form;

use Drupal\a12s_maps_sync\BatchService;
use Drupal\a12s_maps_sync\Entity\Converter;
use Drupal\a12s_maps_sync\Entity\Profile;
use Drupal\a12s_maps_sync\Exception\MapsException;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;

class ImportObjectForm extends FormBase {

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'a12s_maps_sync.import_object_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $form['intro'] = [
      '#markup' => $this->t('This form allows you to import a single MaPS object / media.'),
    ];

    $profileOptions = [];
    foreach (Profile::loadMultiple() as $profile) {
      $profileOptions[$profile->getPythonProfileId()] = $profile->label();
    }

    $form['#prefix'] = '<div id="import-form-wrapper">';
    $form['#suffix'] = '</div>';

    $form['profile'] = [
      '#type' => 'select',
      '#title' => $this->t('Profile'),
      '#options' => $profileOptions,
      '#description' => $this->t('Specify the profile to use to import the entity.'),
      '#ajax' => [
        'callback' => '::updateConvertersValues',
        'event' => 'change',
        'wrapper' => 'import-form-wrapper',
      ],
      '#empty_option' => $this->t('- Select -'),
      '#required' => TRUE,
    ];

    $userInput = $form_state->getUserInput();
    if (!empty($userInput['profile'])) {
      $converterOptions = [];

      $profile = Profile::load($userInput['profile']);
      foreach ($profile->getConverters() as $converter) {
        $converterOptions[$converter->id()] = $converter->label();
      }

      $form['converter'] = [
        '#type' => 'select',
        '#title' => $this->t('Converter'),
        '#options' => $converterOptions,
        '#description' => $this->t('Specify the converter to use to import the entity.'),
        '#required' => TRUE,
      ];
    }

    $form['maps_type'] = [
      '#type' => 'select',
      '#title' => $this->t('MaPS Type'),
      '#options' => [
        'object' => $this->t('Object'),
#        'media' => $this->t('Media'),
      ],
      '#required' => TRUE,
    ];

    $form['with_dependencies'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('With dependencies'),
      '#default_value' => TRUE,
      '#description' => $this->t('Import the dependencies. It is not advised to uncheck this, as it can lead to errors or unexpected results'),
    ];

    $form['maps_id'] = [
      '#type' => 'textfield',
      '#title' => $this->t('MaPS ID'),
      '#description' => $this->t('The MaPS identifier'),
      '#required' => TRUE,
    ];

    $form['actions'] = ['#type' => 'actions'];
    $form['actions']['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Import'),
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $converter = Converter::load($form_state->getValue('converter'));
    $batch = BatchService::getObjectImportBatchDefinition(
      (int) $form_state->getValue('maps_id'),
      $converter,
      (bool) $form_state->getValue('with_dependencies'),
    );

    if ($batch) {
      batch_set($batch);
    }
    else {
      throw new MapsException("Unable to import the given data.");
    }
  }

  /**
   * @param array $form
   * @param \Drupal\Core\Form\FormStateInterface $formState
   *
   * @return array
   */
  public function updateConvertersValues(array &$form, FormStateInterface $formState): array {
    return $form;
  }

}
