<?php

namespace Drupal\a12s_maps_sync\Form;

use Drupal\a12s_maps_sync\Maps\BaseInterface;
use Drupal\Core\Entity\EntityFieldManager;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Field\BaseFieldDefinition;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;

/**
 * Class SettingsForm.
 */
class SettingsForm extends ConfigFormBase {

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return [
      'a12s_maps_sync.settings',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'settings_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('a12s_maps_sync.settings');

    $default = [
      'taxonomy_term',
      'media',
      'file',
      'content',
    ];

    $entity_types = \Drupal::entityTypeManager()->getDefinitions();
    $options = [];
    foreach ($entity_types as $entity_type) {
      $options[$entity_type->id()] = $entity_type->getLabel();
    }

    $form['allowed_entity_types'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Allowed entity types'),
      '#description' => $this->t('The entity types allowed to be converted through the import.'),
      '#options' => $options,
      '#default_value' => $config->get('allowed_entity_types') ?: $default,
    ];

    $form['ignore_lock'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Ignore the lock management'),
      '#description' => $this->t('Ignore the lock management during import, for whatever reason. It is the same thing as launching every command with the --force option.'),
      '#default_value' => $config->get('ignore_lock') ?: FALSE,
    ];

    $form['import_batch_size'] = [
      '#type' => 'number',
      '#title' => $this->t('Import batch size'),
      '#description' => $this->t('The number of objects imported in each batch process.'),
      '#default_value' => $config->get('import_batch_size') ?: 100,
    ];

    $form['batch_max_iterations_count'] = [
      '#type' => 'number',
      '#title' => $this->t('Batch max iterations count'),
      '#description' => $this->t('The max iterations count for Drush batch processing.'),
      '#default_value' => $config->get('batch_max_iterations_count') ?: 4,
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    parent::submitForm($form, $form_state);

    $config = $this->config('a12s_maps_sync.settings');

    $currentConfig = $config->get('allowed_entity_types') ? array_filter($config->get('allowed_entity_types')) : [];
    $newConfig = $form_state->getValue('allowed_entity_types') ? array_filter($form_state->getValue('allowed_entity_types')) : [];

    $removed = array_diff($currentConfig, $newConfig);
    $added = array_diff($newConfig, $currentConfig);

    $this->config('a12s_maps_sync.settings')
      ->set('allowed_entity_types', $form_state->getValue('allowed_entity_types'))
      ->set('ignore_lock', $form_state->getValue('ignore_lock'))
      ->set('import_batch_size', $form_state->getValue('import_batch_size'))
      ->set('batch_max_iterations_count', $form_state->getValue('batch_max_iterations_count'))
      ->save();

    foreach ($removed as $entityTypeId) {
      \Drupal::messenger()->addWarning($this->t(
        "The %entity_type has been disallowed, but the GID base field still exist.",
        ['%entity_type' => $entityTypeId]
      ));
    }

    if (!empty($added)) {
      $typeManager = \Drupal::entityTypeManager();
      $typeManager->clearCachedDefinitions();

      /** @var EntityFieldManager $fieldManager */
      $fieldManager = \Drupal::service('entity_field.manager');
      $fieldManager->clearCachedFieldDefinitions();

      $operations = [];
      foreach ($added as $entityTypeId) {
        $baseFieldDefinitions = $fieldManager->getBaseFieldDefinitions($entityTypeId);

        if (isset($baseFieldDefinitions[BaseInterface::GID_FIELD])) {
          $entityType = $typeManager->getDefinition($entityTypeId);
          $operations[] = [self::class . '::createGidField', [$entityType, $baseFieldDefinitions[BaseInterface::GID_FIELD]]];
        }
      }

      if (!empty($operations)) {
        $batch = [
          'title' => t('Creating needed a12s MaPS Sync GID fields.'),
          'operations' => $operations,
          'finished' => self::class . '::batchFinished',
        ];

        batch_set($batch);
      }
    }
  }

  /**
   * @param EntityTypeInterface $entityType
   * @param BaseFieldDefinition $definition
   * @return void
   */
  public static function createGidField(EntityTypeInterface $entityType, BaseFieldDefinition $definition) {
    \Drupal::entityDefinitionUpdateManager()->installFieldStorageDefinition(
      BaseInterface::GID_FIELD,
      $entityType->id(),
      'a12s_maps_sync',
      $definition
    );
  }

  /**
   * @return void
   */
  public static function batchFinished() {}

}
