<?php

namespace Drupal\a12s_maps_sync;

use Drupal\a12s_maps_sync\Entity\ConverterInterface;
use Drupal\a12s_maps_sync\Entity\ProfileInterface;
use Drupal\a12s_maps_sync\Exception\MapsApiException;
use Drupal\Component\Serialization\Json;
use Drupal\Core\Utility\Error;
use GuzzleHttp\ClientInterface;
use GuzzleHttp\Exception\GuzzleException;

class MapsApi {

  /**
   * @param \GuzzleHttp\ClientInterface $client
   */
  public function __construct(
    protected ClientInterface $client
  ) {}

  /**
   * @return string
   * @throws \Drupal\a12s_maps_sync\Exception\MapsApiException
   */
  public function getBaseUrl(): string {
    $baseUrl = getenv('A12S_MAPS_SYNC_API_URL');
    if (!$baseUrl) {
      throw new MapsApiException("No base URL defined: check the A12S_MAPS_SYNC_API_URL environment variable");
    }

    return rtrim($baseUrl, '/');
  }

  /**
   * Checks if the API is available by making a HEAD request to the base URL.
   *
   * @return bool
   *   Returns TRUE if the API is available, otherwise FALSE.
   */
  public function isAvailable(): bool {
    try {
      $url = $this->getBaseUrl() . '/_healthcheck';
      $response = $this->client->request('HEAD', $url, [
        'headers' => ['X-Maps-Api-Key' => getenv('A12S_MAPS_SYNC_API_KEY')],
        'timeout' => 0,
      ]);
      return $response->getStatusCode() === 200;
    }
    catch (MapsApiException|GuzzleException $e) {
      Error::logException(\Drupal::logger('a12s_maps_sync'), $e);
    }

    return FALSE;
  }

  /**
   * @param \Drupal\a12s_maps_sync\Entity\ProfileInterface $profile
   * @param array $filters
   * @param bool $ordered
   *
   * @return mixed
   * @throws \Drupal\a12s_maps_sync\Exception\MapsApiException
   * @throws \GuzzleHttp\Exception\GuzzleException
   */
  public function getLinks(ProfileInterface $profile, array $filters = [], bool $ordered = FALSE): mixed {
    // Filter the available filters.
    foreach ($filters as $key => $value) {
      if (!in_array($key, ['source_id', 'target_id', 'type_id'])) {
        unset($filters[$key]);
      }
    }

    $url = $this->getBaseUrl() . '/' . $profile->getPythonProfileId() . '/links';
    $params = ['filters' => $filters, 'ordered' => $ordered];

    return $this->execute($url, $params);
  }

  /**
   * @param \Drupal\a12s_maps_sync\Entity\ProfileInterface $profile
   * @param array $filters
   * @param int $limit
   * @param int $fromTime
   * @param bool $countOnly
   *
   * @return array|int
   * @throws \Drupal\a12s_maps_sync\Exception\MapsApiException
   * @throws \GuzzleHttp\Exception\GuzzleException
   */
  public function getObjects(ProfileInterface $profile, array $filters = [], int $limit = -1, int $fromTime = 0, bool $countOnly = FALSE): array|int {
    $replacements = [
      'object_nature' => 'nature',
      'object_type' => 'type',
    ];

    foreach ($replacements as $source => $target) {
      if (!empty($filters[$source])) {
        $filters[$target] = $filters[$source];
      }
    }

    // Filter the available filters.
    foreach ($filters as $key => $value) {
      if (!in_array($key, ['id', 'parent_id', 'source_id', 'code', 'nature', 'type', 'status', 'class']) && !str_starts_with($key, 'attribute_')) {
        unset($filters[$key]);
      }
    }

    $url = $this->getBaseUrl() . '/' . $profile->getPythonProfileId() . '/objects';

    $params = ['filters' => $filters, 'from_time' => $fromTime];
    if ($countOnly) {
      $params['count_only'] = TRUE;
    }

    if ($limit > 0) {
      $params['limit'] = $limit;
    }

    return $this->execute($url, $params);
  }

  /**
   * @param \Drupal\a12s_maps_sync\Entity\ConverterInterface $converter
   * @param array $filters
   * @param int $limit
   * @param int $fromTime
   * @param bool $countOnly
   *
   * @return array
   * @throws \Drupal\a12s_maps_sync\Exception\MapsApiException
   * @throws \GuzzleHttp\Exception\GuzzleException
   */
  public function getMedias(ConverterInterface $converter, array $filters = [], int $limit = -1, int $fromTime = 0, bool $countOnly = FALSE): array {
    // Some replacement.
    if (!empty($filters['media_type'])) {
      $filters['type'] = $filters['media_type'];
      unset($filters['media_type']);
    }

    // Filter the available filters.
    foreach ($filters as $key => $value) {
      if (!in_array($key, ['id', 'type', 'object_id', 'extension', 'from_time']) && !str_starts_with($key, 'attribute_')) {
        unset($filters[$key]);
      }
    }

    $url = $this->getBaseUrl() . '/' . $converter->getProfile()->getPythonProfileId() . '/medias';
    $params = ['filters' => $filters, 'from_time' => $fromTime];
    if ($countOnly) {
      $params['count_only'] = TRUE;
    }

    if ($limit > 0) {
      $params['limit'] = $limit;
    }

    return $this->execute($url, $params);
  }

  /**
   * @param \Drupal\a12s_maps_sync\Entity\ProfileInterface $profile
   * @param int $attributeId
   * @param int $fromTime
   *
   * @return array
   * @throws \Drupal\a12s_maps_sync\Exception\MapsApiException
   * @throws \GuzzleHttp\Exception\GuzzleException
   */
  public function getLibraries(ProfileInterface $profile, int $attributeId, int $fromTime = 0): array {
    $url = $this->getBaseUrl() . '/' . $profile->getPythonProfileId() . '/libraries/' . $attributeId . '?from_time=' . $fromTime;
    return $this->execute($url);
  }

  /**
   * @param \Drupal\a12s_maps_sync\Entity\ProfileInterface $profile
   * @param array $filters
   *
   * @return array
   * @throws \Drupal\a12s_maps_sync\Exception\MapsApiException
   * @throws \GuzzleHttp\Exception\GuzzleException
   */
  public function getConfiguration(ProfileInterface $profile, array $filters = []): array {
    // Filter the available filters.
    foreach ($filters as $key => $value) {
      if (!in_array($key, ['id', 'code', 'type', 'id_language'])) {
        unset($filters[$key]);
      }
    }

    // Add a default language.
    if (empty($filters['id_language'])) {
      $filters['id_language'] = $profile->getDefaultMapslanguage() ?? 1;
    }

    $url = $this->getBaseUrl() . '/' . $profile->getPythonProfileId() . '/config';
    return $this->execute($url, $filters);
  }

  /**
   * @param ProfileInterface $profile
   * @return array
   * @throws MapsApiException
   * @throws \GuzzleHttp\Exception\GuzzleException
   */
  public function getObjectProperties(ProfileInterface $profile): array {
    $url = $this->getBaseUrl() . '/' . $profile->getPythonProfileId() . '/object-properties';
    return $this->execute($url);
  }

  /**
   * @param string $url
   * @param array $params
   *
   * @return mixed
   * @throws \Drupal\a12s_maps_sync\Exception\MapsApiException
   * @throws \GuzzleHttp\Exception\GuzzleException
   */
  protected function execute(string $url, array $params = []): mixed {
    if (!empty($params)) {
      $url .= '?' . http_build_query($params);
    }

    $response = $this->client->request(
      'GET',
      $url,
      [
        'headers' => ['X-Maps-Api-Key' => getenv('A12S_MAPS_SYNC_API_KEY')],
        'timeout' => 0,
      ]
    );

    $statusCode = $response->getStatusCode();
    if ($statusCode !== 200) {
      throw new MapsApiException("Expected code 200, got $statusCode");
    }
    else {
      $contents = $response->getBody()->getContents();
      return Json::decode($contents);
    }
  }

}
