<?php

namespace Drupal\a12s_maps_sync\Plugin\MapsSyncHandler;

use Drupal\a12s_maps_sync\Entity\ConverterInterface;
use Drupal\a12s_maps_sync\Event\ConverterItemImportEvent;
use Drupal\a12s_maps_sync\Maps\BaseInterface;
use Drupal\a12s_maps_sync\Plugin\MapsSyncHandlerBase;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Language\LanguageInterface;
use Drupal\file\FileInterface;
use Drupal\media\MediaInterface;

/**
 * Class MediaHandler
 *
 * Handler used for files when we want to create the related media on the fly.
 *
 * @MapsSyncHandler(
 *   id = "media",
 *   label = @Translation("Media")
 * )
 */
class MediaHandler extends MapsSyncHandlerBase {

  /**
   * Get the media types mapping.
   *
   * @return array
   */
  public function getMediaTypesMapping(): array {
    return [
      1 => [
        'type' => 'image',
        'field' => 'field_image',
      ],
      2 => [
        'type' => 'document',
        'field' => 'field_document_file',
      ],
    ];
  }

  /**
   * Get the media type mapping for a given object.
   *
   * @param \Drupal\a12s_maps_sync\Maps\BaseInterface $object
   *
   * @return array
   */
  public function getMediaTypeMapping(BaseInterface $object): array {
    return $this->getMediaTypesMapping()[$object->get('type')];
  }

  /**
   * {@inheritdoc}
   */
  public function postSave(EntityInterface $entity, BaseInterface $object, ConverterInterface $converter, LanguageInterface $language = NULL): void {
    // Create the media.
    $media = $this->createMedia($entity, $object, $converter, $language);

    if ($converter->getFlushImageStyles() && $entity instanceof FileInterface) {
      $this->flushImageStyles($entity);
    }

    // We need to retrigger the dispatch of the item import finished event here.
    $event_dispatcher = \Drupal::service('event_dispatcher');
    $event = new ConverterItemImportEvent($converter, $object, $media, $language);
    $event_dispatcher->dispatch($event, ConverterItemImportEvent::FINISHED);
  }

  /**
   * Removes image style derivatives related to the given file.
   *
   * @param \Drupal\file\FileInterface $file
   *   The file entity whose image styles should be flushed.
   */
  protected function flushImageStyles(FileInterface $file): void {
    $originalUri = $file->getFileUri();

    /** @var \Drupal\Core\StreamWrapper\StreamWrapperManagerInterface $streamWrapperManager */
    $streamWrapperManager = \Drupal::service('stream_wrapper_manager');

    $scheme = $streamWrapperManager::getScheme($originalUri);
    $target = $streamWrapperManager::getTarget($originalUri);

    $stylesRoot = 'public://styles';

    if (is_dir($stylesRoot)) {
      // Scan the styles directory to find style folders.
      foreach (scandir($stylesRoot) as $styleName) {
        if ($styleName === '.' || $styleName === '..') {
          continue;
        }

        $stylePath = $stylesRoot . '/' . $styleName;
        if (is_dir($stylePath)) {
          $derivativeUri = $stylePath . '/' . $scheme . '/' . $target;

          if (file_exists($derivativeUri)) {
            unlink($derivativeUri);
          }
        }
      }
    }
  }

  /**
   * Create a media entity from the provided entity and object.
   *
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   The entity to associate with the media.
   * @param \Drupal\a12s_maps_sync\Maps\BaseInterface $object
   *   The object containing data used to create the media.
   * @param \Drupal\a12s_maps_sync\Entity\ConverterInterface $converter
   *   The converter interface for handling mappings and entity resolution.
   * @param \Drupal\Core\Language\LanguageInterface|null $language
   *   (optional) The language interface to define the language context.
   *
   * @return \Drupal\media\MediaInterface
   *   The created media entity.
   */
  protected function createMedia(EntityInterface $entity, BaseInterface $object, ConverterInterface $converter, LanguageInterface $language = NULL): MediaInterface {
    // To find the media type, we are using the media type from MaPS.
    $media_type_mapping = $this->getMediaTypeMapping($object);

    // Check if the entity exists.
    $sourceHandler = $converter->getSourceHandler();
    $media = $sourceHandler->getEntity('media', $media_type_mapping['type'], $object, $converter->getGid(), $language);

    $media->{$media_type_mapping['field']}->target_id = $entity->id();
    $media->set(BaseInterface::GID_FIELD, $entity->get(BaseInterface::GID_FIELD)->value);

    // Specific case for images, we have to set the "alt" property of the image field.
    if ($media_type_mapping['type'] === 'image') {
      $_language = $language !== NULL && $entity->isTranslatable()
        ? \Drupal::config('a12s_maps_sync.languages_mapping')
          ->get($language->getId())
        : $converter->getProfile()->getDefaultMapslanguage();

      $media->{$media_type_mapping['field']}->alt = $object->get('MEDIALIBELLE', $_language);
    }

    $media->save();
    return $media;
  }

  /**
   * @inheritDoc
   */
  public function postDelete(EntityInterface $entity, BaseInterface $object, ConverterInterface $converter, LanguageInterface $language = NULL): void {
    $media_type_mapping = $this->getMediaTypeMapping($object);

    // Check if the entity exists.
    $sourceHandler = $converter->getSourceHandler();
    $media = $sourceHandler->getEntity('media', $media_type_mapping['type'], $object, $converter->getGid(), $language);
    $media->delete();
  }

}
