<?php

namespace Drupal\a12s_maps_sync\Plugin;

use Drupal\a12s_maps_sync\Converter\Mapping;
use Drupal\a12s_maps_sync\Entity\ConverterInterface;
use Drupal\a12s_maps_sync\Entity\ProfileInterface;
use Drupal\a12s_maps_sync\Exception\MappingRequiredException;
use Drupal\a12s_maps_sync\Maps\Exception\MapsEntityDefinitionException;
use Drupal\a12s_maps_sync\Maps\BaseInterface;
use Drupal\a12s_maps_sync\Maps\MapsObject;
use Drupal\Component\Plugin\PluginBase;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Language\LanguageInterface;
use Drupal\Core\Logger\LoggerChannelTrait;

/**
 * Base class for MaPS Sync handler plugins.
 */
abstract class MapsSyncHandlerBase extends PluginBase implements MapsSyncHandlerInterface {

  use LoggerChannelTrait;

  /**
   * {@inheritDoc}
   */
  public function importData(ProfileInterface $profile, array $converters, int $limit = MapsSyncHandlerInterface::QUERY_LIMIT): array {
    $results = [];

    /** @var \Drupal\a12s_maps_sync\Entity\ConverterInterface $converter */
    foreach ($converters as $converter) {
      $results = $converter->import($limit);
    }

    return $results;
  }

  /**
   * {@inheritDoc}
   */
  public function postDelete(EntityInterface $entity, BaseInterface $object, ConverterInterface $converter, LanguageInterface $language = NULL): void {
    // Nothing to do...
  }

  /**
   * {@inheritDoc}
   *
   * @param BaseInterface $object
   * @param ConverterInterface $targetConverter
   * @param LanguageInterface|null $language
   *
   * @return EntityInterface|null
   *
   * @throws MapsEntityDefinitionException
   *   When the converter definition is incomplete.
   * @throws \Drupal\Core\Entity\EntityStorageException
   * @throws \Drupal\a12s_maps_sync\Exception\MappingRequiredException
   */
  public function convertItem(BaseInterface $object, ConverterInterface $targetConverter, LanguageInterface $language = NULL): ?EntityInterface {
    $sourceHandler = $targetConverter->getSourceHandler();

    /** @var \Drupal\a12s_maps_sync\Plugin\MappingHandlerPluginManager $mappingHandlerPluginManager */
    $mappingHandlerPluginManager = \Drupal::service('plugin.manager.maps_sync_mapping_handler');

    if (!$entity = $sourceHandler->getEntity($targetConverter->getConverterEntityType(), $targetConverter->getConverterBundle(), $object, $targetConverter->getGid(), $language)) {
      return NULL;
    }

    // Manage the entity status.
    $status_mgmt = $targetConverter->getStatusManagement() ?? ConverterInterface::STATUS_MANAGEMENT_UNPUBLISH;

    // Check if the object is published.
    $statusValue = match($targetConverter->getStatusProperty() ?? 'status') {
      'status' => $object->get('status'),
      'properties' => $object instanceof MapsObject ? ($object->getProperties()[$targetConverter->getStatusPropertyName()] ?? NULL) : NULL,
      'attributes' => $object->get($targetConverter->getStatusPropertyName(), $targetConverter->getProfile()->getDefaultMapslanguage(), 0),
      default => NULL,
    };

    if ($targetConverter->getMapsType() === 'object') {
      // Special case: only update existing entities, without changing the status.
      if ($status_mgmt === ConverterInterface::STATUS_MANAGEMENT_UPDATE_EXISTING_IGNORE_STATUS) {
        $gid_definition = $targetConverter->getGid();

        if (empty($gid_definition)) {
          throw new MapsEntityDefinitionException($targetConverter->getConverterEntityType(), 'gid');
        }

        // Only update entities, no creation here.
        if ($gid = $sourceHandler->getGid($gid_definition, $object)) {
          $storage = \Drupal::service('entity_type.manager')
            ->getStorage($targetConverter->getConverterEntityType());

          if (empty($storage->loadByProperties([BaseInterface::GID_FIELD => $gid]))) {
            return NULL;
          }
        }
      }
      // Ensure that the entity has a "setPublished" method.
      else {
        // Check if the entity has a published status.
        if (in_array($statusValue, $targetConverter->getPublishedStatuses(), TRUE)) {
          if (method_exists($entity, 'setPublished')) {
            $entity->setPublished();
          }
        }
        // Deleted.
        elseif (in_array($statusValue, $targetConverter->getDeletedStatuses(), TRUE)) {
          $entity->delete();
          $this->postDelete($entity, $object, $targetConverter, $language);
          return NULL;
        }
        // Default case / unpublished case.
        else {
          if (method_exists($entity, 'setUnpublished')) {
            $entity->setUnpublished();
          }
          else {
            $entity->delete();
            $this->postDelete($entity, $object, $targetConverter, $language);
            return NULL;
          }
        }
      }
    }
    elseif ($targetConverter->getMapsType() === 'media') {
      if ($status_mgmt === ConverterInterface::STATUS_MANAGEMENT_DELETE) {
        // Get the status "value".
        if ($targetConverter->getStatusProperty() === 'attributes') {
          if ($object->get(str_replace('attribute_', '', $targetConverter->getStatusPropertyName())) !== $targetConverter->getMediaStatusPublishedValue()) {
            $entity->delete();
            $this->postDelete($entity, $object, $targetConverter, $language);
            return NULL;
          }
        }
      }
    }

    $mapping = $sourceHandler->getFixedMapping($targetConverter->getProfile());
    $mapping = array_merge($mapping, $targetConverter->getMapping());

    if (!empty($mapping)) {
      /** @var \Drupal\a12s_maps_sync\Converter\Mapping $mappingItem */
      foreach ($mapping as $mappingItem) {
        // Manage cases with a format defined (for example: html_full values).
        // @todo Manage this.
        /*
        if (!empty($target['format']) && !empty($target['field_name'])) {
          $format = $target['format'];
          $target = $target['field_name'];

          $entity->get($target)->format = $format;
        }
        */

        $_language = $language !== NULL && $entity->isTranslatable()
          ? \Drupal::config('a12s_maps_sync.languages_mapping')
            ->get($language->getId())
          : $targetConverter->getProfile()->getDefaultMapslanguage();

        if (is_null($_language)) {
          \Drupal::logger('a12s_maps_sync')->warning("No MaPS language configured for the Drupal language %language", ['%language' => $language->getId()]);
          continue;
        }

        $mappingHandler = $mappingHandlerPluginManager->createInstance($mappingItem->getHandlerId());
        $values = $mappingHandler->mapData($entity, $object, $mappingItem, $_language, $language);

        if (empty($values) && $mappingItem->isRequired()) {
          if ($mappingItem->getRequiredBehavior() === Mapping::MAPPING_REQUIRED_ERROR) {
            throw new MappingRequiredException("Missing required value for {$mappingItem->getSource()} on {$targetConverter->getMapsType()} {$object->getId()}");
          }
          else {
            return NULL;
          }
        }
      }
    }

    return $entity;
  }

  /**
   * {@inheritdoc}
   */
  public function preSave(EntityInterface $entity, BaseInterface $object, ConverterInterface $converter, LanguageInterface $language = NULL) {
    // If the entity is translatable and the label empty, we try to use
    // the label of the default language.
    if ($entity->isTranslatable() && empty($entity->label())) {
      $default = $entity->getTranslation(\Drupal::languageManager()
        ->getDefaultLanguage()
        ->getId());
      $label = $default->label();

      $entity->set($entity->getEntityType()->getKey('label'), $label);
    }
  }

  /**
   * {@inheritdoc}
   */
  public function postSave(EntityInterface $entity, BaseInterface $object, ConverterInterface $converter, LanguageInterface $language = NULL) {
  }

}
