<?php

namespace Drupal\a12s_maps_sync\Plugin\SourceHandler;

use Drupal\a12s_maps_sync\Entity\ConverterInterface;
use Drupal\a12s_maps_sync\Exception\MapsApiException;
use Drupal\a12s_maps_sync\MapsApiTrait;
use Drupal\a12s_maps_sync\Plugin\SourceHandlerPluginBase;
use Drupal\Core\Utility\Error;
use GuzzleHttp\Exception\GuzzleException;

/**
 * Plugin implementation of the a12s_maps_sync_source_handler.
 *
 * @SourceHandler(
 *   id = "library",
 *   label = @Translation("Library"),
 *   description = @Translation("Library")
 * )
 */
class Library extends SourceHandlerPluginBase {

  use MapsApiTrait;

  /**
   * {@inheritdoc}
   * @param \Drupal\a12s_maps_sync\Entity\ConverterInterface $converter
   */
  public function getFilterTypes(ConverterInterface $converter): array {
    return [
      'library' => t('Library name'),
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getFilterValues(ConverterInterface $converter, string $type): array {
    $profile = $converter->getProfile();

    // Special case for library:
    // we have to retrieve all attributes with the type "library".
    // and since the data is stored in a blob, we have to postprocess the
    // API results.
    try {
      $config = $this->mapsApi()->getConfiguration($profile, ['type' => 'attribute']);
    }
    catch (MapsApiException|GuzzleException $e) {
      Error::logException(\Drupal::logger('a12s_maps_sync'), $e);
      return [];
    }

    foreach ($config as $k => $item) {
      if ($item['data']['attribute_type_code'] !== 'library') {
        unset($config[$k]);
      }
    }

    $options = [];
    foreach ($config as $item) {
      $options[$item['id']] = $item['value'] . " ({$item['code']})";
    }

    return $options;
  }

  /**
   * {@inheritdoc}
   */
  public function getCountData(ConverterInterface $converter, array $filters, int $fromTime): int {
    $data = $this->mapsApi()->getLibraries($converter->getProfile(), $filters['library']['value'], $fromTime);
    return count($data);
  }

  /**
   * {@inheritdoc}
   */
  public function getData(ConverterInterface $converter, array $filters, int $limit, int $fromTime = 0): array {
    if (empty($filters['library'])) {
      throw new MapsApiException("Missing library attribute id.");
    }

    // @todo: make use of limit.

    $data = $this->mapsApi()->getLibraries($converter->getProfile(), $filters['library']['value'], $fromTime);

    $return = [];
    foreach ($data as $datum) {
      $library = new \Drupal\a12s_maps_sync\Maps\Library(
        $datum['id'],
        $datum['profile'],
        $datum['code'],
        $datum['id_attribute'],
        $datum['value'],
        (int) $datum['imported'],
      );

      $library->setProfile($converter->getProfile()->id());
      $library->setConverter($converter->id());

      $return[] = $library;
    }

    return $return;
  }

  /**
   * {@inheritdoc}
   */
  public function hasAttributes(): bool {
    return FALSE;
  }

  /**
   * {@inheritdoc}
   */
  public function getMappingSources(): array {
    return ['value' => $this->t('Value')];
  }

  /**
   * {@inheritdoc}
   */
  public function getAllowedGidEntityKeys(): array {
    return ['id', 'code'];
  }

  /**
   * {@inheritdoc}
   */
  public function isAttributable(): bool {
    return FALSE;
  }

}
