<?php

namespace Drupal\a12s_maps_sync\Plugin\SourceHandler;

use Drupal\a12s_maps_sync\Converter\Mapping;
use Drupal\a12s_maps_sync\Entity\ConverterInterface;
use Drupal\a12s_maps_sync\Entity\ProfileInterface;
use Drupal\a12s_maps_sync\Plugin\SourceHandlerPluginBase;

/**
 * Plugin implementation of the a12s_maps_sync_source_handler.
 *
 * @SourceHandler(
 *   id = "media",
 *   label = @Translation("Media"),
 *   description = @Translation("Media.")
 * )
 */
class Media extends SourceHandlerPluginBase {

  use AttributesTrait;

  /**
   * {@inheritdoc}
   */
  public function getFilterTypes(ConverterInterface $converter): array {
    $filterTypes = [
      'Properties' => [
        'media_type' => $this->t('Media type'),
        'extension' => $this->t('Extension'),
      ],
    ];

    $filterTypes['Attributes'] = $this->getAttributesFilters($converter);

    return $filterTypes;
  }

  /**
   * {@inheritdoc}
   */
  public function getCountData(ConverterInterface $converter, array $filters, int $fromTime): int {
    $data = $this->mapsApi()->getMedias($converter, $filters, -1, $fromTime, TRUE);
    return count($data);
  }

  /**
   * {@inheritdoc}
   */
  public function getData(ConverterInterface $converter, array $filters, int $limit, int $fromTime = 0): array {
    $data = $this->mapsApi()->getMedias($converter, $filters, $limit, $fromTime);

    $return = [];
    foreach ($data as $datum) {
      $media = new \Drupal\a12s_maps_sync\Maps\Media(
        $datum['id'],
        $datum['extension'],
        $datum['url'],
        $datum['filename'],
        $datum['type'],
        $datum['weight'],
        (int) $datum['imported'],
      );

      $media->setProfile($converter->getProfile()->id());
      $media->setConverter($converter->id());

      // Add the attributes.
      $attributes = MapsObject::generateAttributes($converter->getProfile(), $datum);

      $media->setAttributes($attributes);
      $return[] = $media;
    }

    return $return;
  }

  /**
   * @param \Drupal\a12s_maps_sync\Entity\ProfileInterface $profile
   *
   * @return array
   */
  public function getFixedMapping(ProfileInterface $profile): array {
    return [
      new Mapping('filename', 'filename', 'default'),
      new Mapping('const:' . $profile->getMediasPath() . '/+filename', 'uri', 'default'),
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getMappingSources(): array {
    return [
      'Properties' => [
        'id' => $this->t('ID'),
        'filename' => $this->t('Filename'),
        'url' => $this->t('Url'),
        'extension' => $this->t('Extension'),
      ],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getAllowedGidEntityKeys(): array {
    return ['id', 'code'];
  }

}
