<?php

namespace Drupal\a12s_maps_sync;

use Drupal\a12s_maps_sync\Exception\StateQueueItemAlreadyInBatchException;
use Drupal\Component\Serialization\PhpSerialize;

final class State extends PhpSerialize {

  private const DRUPAL_STATE_KEY = 'a12s_maps_sync_process';

  /**
   * The queue.
   *
   * @var \Drupal\a12s_maps_sync\StateQueueItem[]
   */
  private array $queue = [];

  /**
   * The batch info.
   *
   * @var \Drupal\a12s_maps_sync\StateBatch|null
   */
  private ?StateBatch $batch = NULL;

  /**
   * @return \Drupal\a12s_maps_sync\StateBatch|null
   */
  public function getBatch(): ?StateBatch {
    return $this->batch;
  }

  /**
   * @param \Drupal\a12s_maps_sync\StateBatch $batch
   *
   * @return $this
   */
  public function setBatch(StateBatch $batch): State {
    $this->batch = $batch;
    return $this;
  }

  /**
   * @return $this
   */
  public function unsetBatch(): State {
    $this->batch = NULL;
    return $this;
  }

  /**
   * @return \Drupal\a12s_maps_sync\StateQueueItem[]
   */
  public function getQueue(): array {
    return $this->queue;
  }

  /**
   * @param \Drupal\a12s_maps_sync\StateQueueItem $queueItem
   * @param bool $forceAdd
   *   Add the item to the queue, even if a similar batch already exists.
   *
   * @return \Drupal\a12s_maps_sync\State
   * @throws \Drupal\a12s_maps_sync\Exception\StateQueueItemAlreadyInBatchException
   */
  public function addToQueue(StateQueueItem $queueItem, bool $forceAdd = FALSE): State {
    $key = $queueItem->getQueueKey();

    // Check if the item is already in the batch.
    if (!$forceAdd) {
      $batch = $this->getBatch();
      if (!is_null($batch) && $batch->getProfile() === $queueItem->getProfile() && $batch->getConverter() === $queueItem->getConverter()) {
        throw new StateQueueItemAlreadyInBatchException();
      }
    }

    // Add or replace in the queue.
    $this->queue[$key] = $queueItem;

    // Simplify the queue if no converter defined.
    if (is_null($queueItem->getConverter())) {
      foreach ($this->queue as $key => $item) {
        // Remove item if it is for the same profile, but for a specific converter.
        if ($item->getProfile() === $queueItem->getProfile() && !is_null($item->getConverter())) {
          unset($this->queue[$key]);
        }
      }
    }

    return $this;
  }

  /**
   * @param \Drupal\a12s_maps_sync\StateQueueItem $queueItem
   *
   * @return $this
   */
  public function removeFromQueue(StateQueueItem $queueItem): State {
    unset($this->queue[$queueItem->getQueueKey()]);
    return $this;
  }

  /**
   * Get the data from the Drupal state.
   *
   * @return \Drupal\a12s_maps_sync\State
   */
  public static function load(): State {
    $drupalState = \Drupal::state()->get(self::DRUPAL_STATE_KEY);
    return $drupalState ? self::decode($drupalState) : new State();
  }

  /**
   * Save the data in Drupal state.
   *
   * @return $this
   */
  public function save(): State {
    \Drupal::state()->set(self::DRUPAL_STATE_KEY, self::encode($this));
    return $this;
  }

  /**
   * Convert to an array.
   * Useful for displaying or debugging.
   *
   * @return string[]
   */
  public function toArray(): array {
    $data = [];

    if ($batch = $this->getBatch()) {
      $data['batch'] = $batch->toArray();
    }

    if (!empty($this->queue)) {
      foreach ($this->queue as $queueItem) {
        $data['queue'][] = $queueItem->toArray();
      }
    }

    return $data;
  }

  /**
   * Flush the queue.
   *
   * @return $this
   */
  public function flushQueue(): State {
    $this->queue = [];
    return $this;
  }

}
