<?php

namespace Drupal\aadhaar_number_widget\Plugin\Field\FieldWidget;

use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Field\WidgetBase;
use Drupal\Core\Form\FormStateInterface;

/**
 * Plugin implementation of the 'aadhaar_number_widget' widget.
 *
 * @FieldWidget(
 *   id = "aadhaar_number_widget",
 *   module = "aadhaar_number_widget",
 *   label = @Translation("Aadhaar Number Widget"),
 *   field_types = {
 *     "string"
 *   }
 * )
 */
class AadhaarNumberWidget extends WidgetBase {

  /**
   * {@inheritdoc}
   */
  public function formElement(FieldItemListInterface $items, $delta, array $element, array &$form, FormStateInterface $form_state): array {

    $element['value'] = $element + [
      '#type' => 'textfield',
      '#default_value' => $items[$delta]->value ?? '',
      '#maxlength' => 14,
      '#element_validate' => [
        [static::class, 'validate'],
      ],
    ];

    return $element;
  }

  /**
   * Validates the Aadhaar number format and checksum.
   *
   * @param array $element
   *   The form element.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   */
  public static function validate(array &$element, FormStateInterface $form_state): void {
    $value = $element['#value'];

    // Skip validation if empty.
    if ($value === '' || $value === NULL) {
      return;
    }

    // Aadhaar number can be in the following formats:
    // - 968415522551
    // - 9684 1552 2551
    // - 9684-1552-2551
    $pattern = "/(^\d{12}$)|(^\d{4}\s\d{4}\s\d{4}$)|(^\d{4}-\d{4}-\d{4}$)/";

    if (preg_match($pattern, $value)) {
      if (!self::isAadhaarValid($value)) {
        $form_state->setError($element, t('The Aadhaar number is not valid.'));
      }
    }
    else {
      $form_state->setError(
        $element,
        t('Invalid Aadhaar number format. Use one of the following:<br>
          968415522551<br>
          9684 1552 2551<br>
          9684-1552-2551')
      );
    }

  }

  /**
   * Checks whether the Aadhaar number is valid using the Verhoeff algorithm.
   *
   * @param string $num
   *   The Aadhaar number (possibly containing spaces or dashes).
   *
   * @return bool
   *   TRUE if valid, FALSE otherwise.
   */
  public static function isAadhaarValid(string $num): bool {
    $num = preg_replace('/[-\s]/', '', $num);
    $expectedDigit = substr($num, -1);
    $actualDigit = self::calculateChecksumDigit(substr($num, 0, -1));

    return $expectedDigit === (string) $actualDigit;
  }

   /**
   * Calculates the Aadhaar checksum digit using the Verhoeff algorithm.
   *
   * @param string $partial
   *   Aadhaar number without the checksum digit.
   *
   * @return int
   *   The checksum digit.
   */
  protected static function calculateChecksumDigit(string $partial): int {
    $dihedral = [
      [0, 1, 2, 3, 4, 5, 6, 7, 8, 9],
      [1, 2, 3, 4, 0, 6, 7, 8, 9, 5],
      [2, 3, 4, 0, 1, 7, 8, 9, 5, 6],
      [3, 4, 0, 1, 2, 8, 9, 5, 6, 7],
      [4, 0, 1, 2, 3, 9, 5, 6, 7, 8],
      [5, 9, 8, 7, 6, 0, 4, 3, 2, 1],
      [6, 5, 9, 8, 7, 1, 0, 4, 3, 2],
      [7, 6, 5, 9, 8, 2, 1, 0, 4, 3],
      [8, 7, 6, 5, 9, 3, 2, 1, 0, 4],
      [9, 8, 7, 6, 5, 4, 3, 2, 1, 0],
    ];

    $permutation = [
      [0, 1, 2, 3, 4, 5, 6, 7, 8, 9],
      [1, 5, 7, 6, 2, 8, 3, 0, 9, 4],
      [5, 8, 0, 3, 7, 9, 6, 1, 4, 2],
      [8, 9, 1, 6, 0, 4, 3, 5, 2, 7],
      [9, 4, 5, 3, 1, 2, 6, 8, 7, 0],
      [4, 2, 8, 6, 5, 7, 3, 9, 0, 1],
      [2, 7, 9, 3, 8, 0, 6, 4, 1, 5],
      [7, 0, 4, 6, 9, 1, 3, 2, 5, 8],
    ];

    $inverse = [0, 4, 3, 2, 1, 5, 6, 7, 8, 9];
    $partial = strrev($partial);

    $digitIndex = 0;
    for ($i = 0, $len = strlen($partial); $i < $len; $i++) {
      $digit = (int) $partial[$i];
      $digitIndex = $dihedral[$digitIndex][$permutation[($i + 1) % 8][$digit]];
    }

    return $inverse[$digitIndex];
  }

}
