/**
 * @file
 * Tracking code suggestions behavior for A/B test paragraph edit forms.
 *
 * Analyzes content in variant fields and generates tracking code suggestions
 * based on detected links. Supports multiple A/B test paragraphs on the same page.
 */
(function (Drupal, drupalSettings, once) {

  'use strict';

  /**
   * Drupal behavior for tracking code suggestions.
   *
   * @type {Drupal~behavior}
   */
  Drupal.behaviors.trackingCodeSuggestions = {
    attach: function (context) {
      // Only run on the main document, not on AJAX-loaded content.
      if (context !== document && !context.querySelector('[data-drupal-selector*="field-variant-a"]')) {
        return;
      }

      // Determine analytics provider from Drupal settings.
      var provider = 'matomo';
      if (drupalSettings.abParagraphsSettings && drupalSettings.abParagraphsSettings.analyticsProvider) {
        provider = drupalSettings.abParagraphsSettings.analyticsProvider;
      }

      // Debounce timer.
      var debounceTimer = null;
      var DEBOUNCE_DELAY = 500;

      // Loader timeout - only show loader if processing takes longer than this.
      var LOADER_DELAY = 150;

      /**
       * Simple hash function for content comparison.
       *
       * @param {string} str
       *   The string to hash.
       *
       * @return {number}
       *   A simple hash number.
       */
      function simpleHash(str) {
        var hash = 0;
        if (!str || str.length === 0) {
          return hash;
        }
        for (var i = 0; i < str.length; i++) {
          var char = str.charCodeAt(i);
          hash = ((hash << 5) - hash) + char;
          hash = hash & hash;
        }
        return hash;
      }

      /**
       * Retrieve the content from a wrapper element.
       *
       * @param {Element} wrapper
       *   The wrapper element to extract content from.
       *
       * @return {string}
       *   The extracted content.
       */
      /**
       * Extract entity ID from autocomplete value like "Title (123)".
       *
       * @param {string} value
       *   The autocomplete input value.
       *
       * @return {string|null}
       *   The entity URL or null if not found.
       */
      function extractEntityUrl(value) {
        if (!value) {
          return null;
        }
        // Match patterns like "Title (123)" or "Title (123) [extra]"
        var match = value.match(/\((\d+)\)(?:\s*\[.*\])?$/);
        if (match && match[1]) {
          return '/node/' + match[1];
        }
        return null;
      }

      function getContentFromWrapper(wrapper) {
        // Find ALL link fields (including nested paragraphs).
        var linkFields = wrapper.querySelectorAll('.field--type-link');
        var links = [];

        linkFields.forEach(function (linkField) {
          // Only process visible link fields.
          var fieldWrapper = linkField.closest('[data-drupal-states]');
          if (fieldWrapper && fieldWrapper.style.display === 'none') {
            return;
          }

          var input = linkField.querySelector('input[type="text"]');
          if (input && input.value.trim()) {
            var originalValue = input.getAttribute('data-resolved-path');
            if (originalValue && originalValue.trim() !== '') {
              links.push(originalValue.trim());
              return;
            }
            // Check if it's an entity autocomplete field.
            if (input.hasAttribute('data-autocomplete-path')) {
              var entityUrl = extractEntityUrl(input.value);
              if (entityUrl) {
                links.push(entityUrl);
                return;
              }
            }
            // Check if input value looks like a URL.
            var inputValue = input.value.trim();
            if (inputValue && /^(https?:\/\/|\/|\<)/i.test(inputValue)) {
              links.push(inputValue);
            }
          }
        });

        if (links.length > 0) {
          return links.join(' ');
        }

        // CKEditor 5.
        var editableEl = wrapper.querySelector('.ck-editor__editable');
        if (editableEl) {
          return editableEl.innerHTML;
        }

        // Textarea / input.
        var field = wrapper.querySelector('textarea, input');
        if (field) {
          return field.value;
        }

        // Fallback: full HTML.
        return wrapper.innerHTML;
      }

      /**
       * Parse an HTML string and return all hrefs.
       *
       * @param {string} html
       *   The HTML string to parse.
       *
       * @return {string[]}
       *   Array of href values.
       */
      function getLinksFromHTML(html) {
        // Quick check - if no 'href' in string, skip parsing.
        if (html.indexOf('href') === -1 && !/^(https?:\/\/|\/)/i.test(html.trim())) {
          return [];
        }

        var tempDiv = document.createElement('div');
        tempDiv.innerHTML = html;
        var anchors = tempDiv.querySelectorAll('a[href]');
        var hrefs = [];

        for (var i = 0; i < anchors.length; i++) {
          var href = anchors[i].getAttribute('href');
          if (href && href.indexOf('javascript:') !== 0) {
            hrefs.push(href);
          }
        }

        // If no anchors but the whole string looks like a URL or path.
        if (hrefs.length === 0) {
          var t = html.trim();
          if (/^(https?:\/\/|\/)/i.test(t)) {
            hrefs.push(t);
          }
        }
        return hrefs;
      }

      /**
       * Generate tracking code for a link based on the analytics provider.
       *
       * @param {string} href
       *   The link URL.
       * @param {string} uuid
       *   The unique ID of the A/B test.
       * @param {string} variantName
       *   The variant name (A or B).
       *
       * @return {string}
       *   The generated tracking code.
       */
      function generateTrackingCode(href, uuid, variantName) {
        if (provider === 'google_analytics') {
          return "gtag('event', 'click', {" +
            "'event_category':'A/B Test'," +
            "'event_label':'" + href + " - " + uuid + " - Variant " + variantName + "'" +
            "});";
        }

        return "_paq.push(['trackEvent', 'A/B Test', 'Click', '" +
          href + " - " + uuid + " - Variant " + variantName +
          "']);";
      }

      /**
       * Show loading indicator in target container.
       *
       * @param {Element} container
       *   The container element to show the loader in.
       */
      function showLoader(container) {
        if (container) {
          container.innerHTML = '<div class="tracking-suggestions-loading">' +
            '<span class="tracking-suggestions-throbber"></span> ' +
            'Loading suggestions...</div>';
        }
      }

      /**
       * Render suggestions for a single variant within a specific container.
       *
       * @param {object} variant
       *   The variant configuration object.
       * @param {string} uuid
       *   The unique ID of the A/B test.
       * @param {Element} container
       *   The A/B test paragraph subform container.
       * @param {object} contentCache
       *   Content cache object for this container.
       */
      function renderVariantSuggestions(variant, uuid, container, contentCache) {
        // Find wrapper within this specific container.
        var wrapper = container.querySelector(variant.wrapperSelector);
        if (!wrapper) {
          return;
        }

        var content = getContentFromWrapper(wrapper);
        var contentHash = simpleHash(content);
        var cacheKey = variant.name.toLowerCase();

        // Skip if content hasn't changed.
        if (contentCache[cacheKey] === contentHash) {
          return;
        }
        contentCache[cacheKey] = contentHash;

        // Find target container within this specific container.
        var targetContainer = container.querySelector(
          variant.targetFieldSelector + ' .form-item__description'
        );

        if (!targetContainer) {
          return;
        }

        if (!content) {
          targetContainer.innerHTML = '<em>No content in variant ' + variant.name + '</em>';
          return;
        }

        var links = getLinksFromHTML(content);

        if (links.length === 0) {
          targetContainer.innerHTML = '<em>No links found in variant ' + variant.name + '</em>';
          return;
        }

        // Build suggestions.
        var suggestions = links.map(function (href) {
          return {
            code: generateTrackingCode(href, uuid, variant.name),
            href: href
          };
        });

        // Enforce max-width.
        targetContainer.style.maxWidth = '100%';

        // Render header + refresh button + each suggestion.
        var header = '<strong>Suggestions (' + suggestions.length + '):</strong> ' +
          '<button type="button" class="refresh-suggestions button button--small">Refresh</button>';

        var items = suggestions.map(function (s) {
          return '<div class="tracking-suggestion">' +
            '<code>' + Drupal.checkPlain(s.code) + '</code> ' +
            '<button type="button" class="add-suggestion button button--small" ' +
            'data-code="' + encodeURIComponent(s.code) + '">Copy</button>' +
            '</div>';
        }).join('');

        targetContainer.innerHTML = header + items;

        // Copy-to-clipboard handlers using event delegation.
        targetContainer.addEventListener('click', function (e) {
          var btn = e.target.closest('.add-suggestion');
          if (btn) {
            e.preventDefault();
            var code = decodeURIComponent(btn.dataset.code);
            if (navigator.clipboard && navigator.clipboard.writeText) {
              navigator.clipboard.writeText(code).then(function () {
                btn.textContent = 'Copied!';
                setTimeout(function () {
                  btn.textContent = 'Copy';
                }, 1500);
              });
            }
          }

          if (e.target.closest('.refresh-suggestions')) {
            e.preventDefault();
            contentCache[cacheKey] = null;
            showSuggestions();
          }
        });
      }

      /**
       * Find all A/B test paragraph containers on the page.
       *
       * @return {Element[]}
       *   Array of A/B test container elements.
       */
      function findAbTestContainers() {
        // Find all variant-a wrapper elements specifically (ending with -wrapper).
        // This avoids matching nested paragraph elements like -field-variant-a-0-subform.
        var variantAWrappers = document.querySelectorAll('[data-drupal-selector$="-field-variant-a-wrapper"]');
        var containers = [];

        variantAWrappers.forEach(function (wrapper) {
          // Find the parent A/B test subform container.
          // Try with paragraphs-nested class first, fallback to paragraphs-subform.
          var container = wrapper.closest('.paragraphs-nested.paragraphs-subform');
          if (!container) {
            container = wrapper.closest('.paragraphs-subform');
          }

          if (container && containers.indexOf(container) === -1) {
            containers.push(container);
          }
        });

        return containers;
      }

      /**
       * Process a single A/B test container.
       *
       * @param {Element} container
       *   The A/B test paragraph subform container.
       * @param {object} containerCache
       *   Content cache object for this container.
       */
      function processAbTestContainer(container, containerCache) {
        // Find UUID field within this container.
        var uuidField = container.querySelector('[data-drupal-selector*="field-unique-id"][data-drupal-selector$="-value"]');
        var uuid = uuidField ? uuidField.value.trim() : '';

        var variants = [
          {
            name: 'A',
            wrapperSelector: '[data-drupal-selector*="field-variant-a"]',
            targetFieldSelector: '[data-drupal-selector*="field-tracking-code-a"]'
          },
          {
            name: 'B',
            wrapperSelector: '[data-drupal-selector*="field-variant-b"]',
            targetFieldSelector: '[data-drupal-selector*="field-tracking-code-b"]'
          }
        ];

        // Show loaders only if we expect processing to take time.
        var loaderTimers = [];
        variants.forEach(function (variant) {
          var targetContainer = container.querySelector(
            variant.targetFieldSelector + ' .form-item__description'
          );
          if (targetContainer) {
            var timer = setTimeout(function () {
              showLoader(targetContainer);
            }, LOADER_DELAY);
            loaderTimers.push(timer);
          }
        });

        // Use requestAnimationFrame to avoid blocking UI.
        requestAnimationFrame(function () {
          // Clear loader timers if we finish quickly.
          loaderTimers.forEach(function (timer) {
            clearTimeout(timer);
          });

          variants.forEach(function (variant) {
            renderVariantSuggestions(variant, uuid, container, containerCache);
          });
        });
      }

      // Store content caches per container (keyed by container element).
      var containerCaches = new WeakMap();

      /**
       * Build and render tracking code suggestions for all A/B test paragraphs.
       */
      function showSuggestions() {
        var containers = findAbTestContainers();

        containers.forEach(function (container) {
          // Get or create cache for this container.
          if (!containerCaches.has(container)) {
            containerCaches.set(container, { a: null, b: null });
          }
          var cache = containerCaches.get(container);

          processAbTestContainer(container, cache);
        });
      }

      /**
       * Debounced version of showSuggestions.
       */
      function debouncedShowSuggestions() {
        if (debounceTimer) {
          clearTimeout(debounceTimer);
        }
        debounceTimer = setTimeout(showSuggestions, DEBOUNCE_DELAY);
      }

      /**
       * Watch for changes to regenerate suggestions.
       */
      function attachObservers() {
        // CKEditor 5 - use a single observer with debouncing.
        context.querySelectorAll('.ck-editor__editable').forEach(function (editable) {
          once('ckeditor-mutation-observer', editable).forEach(function (el) {
            var mo = new MutationObserver(debouncedShowSuggestions);
            mo.observe(el, {childList: true, subtree: true, characterData: true});
          });
        });

        // Other fields - only on blur with debouncing.
        once('suggestion-listeners', 'textarea, input, select', context).forEach(function (el) {
          el.addEventListener('blur', debouncedShowSuggestions);
        });
      }

      // Initialize.
      showSuggestions();
      attachObservers();
    }
  };

})(Drupal, drupalSettings, once);
