<?php

namespace Drupal\ab_paragraphs\Form;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\TypedConfigManagerInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Url;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Configure A/B Paragraphs settings.
 */
class AbParagraphsSettingsForm extends ConfigFormBase {

  /**
   * The module handler service.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface
   */
  protected ModuleHandlerInterface $moduleHandler;

  /**
   * Constructs an AbParagraphsSettingsForm object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The factory for configuration objects.
   * @param \Drupal\Core\Config\TypedConfigManagerInterface $typed_config_manager
   *   The typed config manager.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $module_handler
   *   The module handler service.
   */
  public function __construct(ConfigFactoryInterface $config_factory, TypedConfigManagerInterface $typed_config_manager, ModuleHandlerInterface $module_handler) {
    parent::__construct($config_factory, $typed_config_manager);
    $this->moduleHandler = $module_handler;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('config.factory'),
      $container->get('config.typed'),
      $container->get('module_handler')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'ab_paragraphs_settings_form';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return ['ab_paragraphs.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $config = $this->config('ab_paragraphs.settings');

    // Build options with status indicators for installed modules.
    $options = [];

    // Check Matomo module.
    $matomo_config = $this->configFactory->get('matomo.settings');
    $matomo_url = $matomo_config->get('url_https') ?: $matomo_config->get('url_http');
    if ($matomo_url) {
      $options['matomo'] = $this->t('Matomo (configured: @url)', ['@url' => $matomo_url]);
    }
    else {
      $options['matomo'] = $this->t('Matomo (not configured)');
    }

    // Check Piwik Pro module.
    $piwik_pro_config = $this->configFactory->get('piwik_pro.settings');
    $piwik_pro_url = $piwik_pro_config->get('piwik_domain');
    if ($piwik_pro_url) {
      $options['piwik_pro'] = $this->t('Piwik Pro (configured: @url)', ['@url' => $piwik_pro_url]);
    }
    else {
      $options['piwik_pro'] = $this->t('Piwik Pro (not configured)');
    }

    // Google Analytics (no URL to retrieve).
    $options['google_analytics'] = $this->t('Google Analytics');

    $form['analytics_provider'] = [
      '#type' => 'radios',
      '#title' => $this->t('Select Analytics Provider'),
      '#description' => $this->t('Choose which analytics provider to use for A/B test tracking. The tracking URL will be automatically retrieved from the selected module configuration.'),
      '#options' => $options,
      '#default_value' => $config->get('analytics_provider') ?: 'matomo',
    ];

    // Show configuration links.
    $form['config_links'] = [
      '#type' => 'details',
      '#title' => $this->t('Configure Analytics Modules'),
      '#open' => FALSE,
    ];

    if ($this->moduleHandler->moduleExists('matomo')) {
      $form['config_links']['matomo'] = [
        '#type' => 'link',
        '#title' => $this->t('Configure Matomo'),
        '#url' => Url::fromRoute('matomo.admin_settings_form'),
        '#prefix' => '<p>',
        '#suffix' => '</p>',
      ];
    }
    else {
      $form['config_links']['matomo'] = [
        '#markup' => '<p>' . $this->t('Matomo module is not enabled.') . '</p>',
      ];
    }

    if ($this->moduleHandler->moduleExists('piwik_pro')) {
      $form['config_links']['piwik_pro'] = [
        '#type' => 'link',
        '#title' => $this->t('Configure Piwik Pro'),
        '#url' => Url::fromRoute('piwik_pro.admin_settings_form'),
        '#prefix' => '<p>',
        '#suffix' => '</p>',
      ];
    }
    else {
      $form['config_links']['piwik_pro'] = [
        '#markup' => '<p>' . $this->t('Piwik Pro module is not enabled.') . '</p>',
      ];
    }

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $this->config('ab_paragraphs.settings')
      ->set('analytics_provider', $form_state->getValue('analytics_provider'))
      ->save();

    parent::submitForm($form, $form_state);
  }

}
