/**
 * @file
 * A/B Test paragraph behavior.
 *
 * Handles variant selection and analytics tracking for A/B test paragraphs.
 * Variant selection is client-side via sessionStorage, allowing full page caching.
 */
(function ($, Drupal, drupalSettings, once) {

  'use strict';

  /**
   * Debug mode flag - set to true to enable console logging.
   *
   * @type {boolean}
   */
  var debugMode = drupalSettings.abParagraphsSettings && drupalSettings.abParagraphsSettings.debug;

  /**
   * Log message to console if debug mode is enabled.
   *
   * @param {string} message
   *   The message to log.
   * @param {*} [data]
   *   Optional data to log.
   */
  function debugLog(message, data) {
    if (debugMode && typeof console !== 'undefined') {
      if (data !== undefined) {
        console.log('[AB Test] ' + message, data);
      }
      else {
        console.log('[AB Test] ' + message);
      }
    }
  }

  /**
   * Check if analytics consent is given via one of known systems.
   *
   * @return {boolean}
   *   True if consent is given or unknown, false if explicitly denied.
   */
  function hasAnalyticsConsent() {
    // Cookiebot.
    if (typeof Cookiebot !== 'undefined') {
      return Cookiebot.consents && Cookiebot.consents.given && Cookiebot.consents.statistics;
    }

    // Klaro!
    if (typeof Klaro !== 'undefined') {
      var manager = Klaro.getManager ? Klaro.getManager() : null;
      return manager && manager.consent && manager.consent.analytics === true;
    }

    // CookieConsent v3 / Osano.
    if (typeof window.CookieConsent !== 'undefined') {
      var cc = window.CookieConsent;
      if (typeof cc.getUserPreferences === 'function') {
        var prefs = cc.getUserPreferences();
        if (prefs && prefs.categories && typeof prefs.categories.includes === 'function') {
          return prefs.categories.includes('analytics');
        }
        return prefs && prefs.analytics === true;
      }
    }

    // Custom system.
    if (window.myCookieConsent && window.myCookieConsent.analyticsAllowed) {
      return true;
    }

    // Default: assume consent (site-wide consent handling).
    return true;
  }

  /**
   * Track an event using the appropriate analytics provider.
   *
   * Supports Matomo (_paq), Piwik Pro (_paq or ppms), and Google Analytics (gtag).
   *
   * @param {string} category
   *   Event category.
   * @param {string} action
   *   Event action.
   * @param {string} label
   *   Event label.
   * @param {number} value
   *   Event value.
   *
   * @return {boolean}
   *   True if event was tracked, false otherwise.
   */
  function trackEvent(category, action, label, value) {
    // Matomo and Piwik Pro both use _paq.
    if (typeof _paq !== 'undefined') {
      _paq.push(['trackEvent', category, action, label, value]);
      debugLog('Tracked via _paq: ' + category + ' / ' + action + ' / ' + label);
      return true;
    }

    // Piwik Pro Tag Manager alternative (ppms).
    if (typeof ppms !== 'undefined' && ppms.cm && typeof ppms.cm.api === 'function') {
      ppms.cm.api('trackEvent', {
        category: category,
        action: action,
        name: label,
        value: value
      });
      debugLog('Tracked via ppms: ' + category + ' / ' + action + ' / ' + label);
      return true;
    }

    // Google Analytics (gtag).
    if (typeof gtag !== 'undefined') {
      gtag('event', action, {
        'event_category': category,
        'event_label': label,
        'value': value
      });
      debugLog('Tracked via gtag: ' + category + ' / ' + action + ' / ' + label);
      return true;
    }

    debugLog('No analytics tracker found');
    return false;
  }

  /**
   * Check if any analytics tracker is available.
   *
   * @return {boolean}
   *   True if a tracker is available.
   */
  function hasAnalyticsTracker() {
    return typeof _paq !== 'undefined' ||
           (typeof ppms !== 'undefined' && ppms.cm && typeof ppms.cm.api === 'function') ||
           typeof gtag !== 'undefined';
  }

  /**
   * Drupal behavior for A/B test paragraphs.
   *
   * @type {Drupal~behavior}
   */
  Drupal.behaviors.abTest = {
    attach: function (context) {
      once('ab-test', '.ab-test-wrapper', context).forEach(function (wrapper) {
        var $wrapper = $(wrapper);
        var uuid = $wrapper.data('uuid');
        var distribution = $wrapper.data('distribution') || '50/50';
        var sessionKey = 'ab_variant_' + uuid;

        debugLog('Initializing A/B test: ' + uuid);

        // Get or determine variant.
        var variant = sessionStorage.getItem(sessionKey);
        if (!variant) {
          var parts = distribution.split('/');
          var threshold = parseInt(parts[0], 10);
          var rand = Math.random() * 100;
          variant = rand < threshold ? 'a' : 'b';
          sessionStorage.setItem(sessionKey, variant);
          debugLog('New variant selected: ' + variant.toUpperCase() + ' (random: ' + rand.toFixed(2) + ', threshold: ' + threshold + ')');
        }
        else {
          debugLog('Variant from session: ' + variant.toUpperCase());
        }

        // Hide both variants and show the selected one.
        $wrapper.find('.ab-variant-a, .ab-variant-b').addClass('hidden-ab-variant');
        var $activeVariant = $wrapper.find('.ab-variant-' + variant);
        $activeVariant.removeClass('hidden-ab-variant');

        // Get tracking code from data attribute.
        var trackingCode = $activeVariant.attr('data-tracking-code') || '';

        // Track "Show" event if consent is given.
        if (hasAnalyticsConsent() && hasAnalyticsTracker()) {
          trackEvent(
            'A/B Test',
            'Show',
            uuid + ' - Variant ' + variant.toUpperCase(),
            1
          );
        }

        // Event delegation for link click tracking.
        if (trackingCode) {
          $activeVariant.on('click', 'a', function () {
            if (!hasAnalyticsConsent() || !hasAnalyticsTracker()) {
              return;
            }

            var href = $(this).attr('href') || '';

            // Track the click event.
            trackEvent(
              'A/B Test',
              'Click',
              href + ' - ' + uuid + ' - Variant ' + variant.toUpperCase(),
              1
            );
          });
          debugLog('Link click tracking attached for variant ' + variant.toUpperCase());
        }

        debugLog('A/B test initialized: ' + uuid + ' - Variant ' + variant.toUpperCase());
      });
    }
  };

})(jQuery, Drupal, drupalSettings, once);
