<?php

namespace Drupal\access_code\Controller;

use Drupal\access_code\Service\AccessCodeManager;
use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Database\Connection;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\Url;
use Drupal\user\Entity\User;
use Drupal\user\UserFloodControl;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpKernel\Exception\AccessDeniedHttpException;

/**
 * Class UseCodeController.
 */
class UseCodeController extends ControllerBase {

  /**
   * @var \Drupal\Core\Database\Connection
   */
  protected $database;

  /**
   * @var \Drupal\Core\Logger\LoggerChannelInterface
   */
  protected $logger;

  /**
   * @var \Drupal\Core\Messenger\Messenger
   */
  protected $messenger;

  /**
   * @var \Drupal\access_code\Service\AccessCodeManager
   */
  protected $accessCodeManager;

  /**
   * @var \Drupal\user\UserFloodControl
   */
  protected $userFloodControl;

  /**
   * Constructor.
   */
  public function __construct(LoggerChannelFactoryInterface $logger_factory, Connection $database, MessengerInterface $messenger, AccessCodeManager $manager, UserFloodControl $user_flood_control) {
    $this->logger = $logger_factory->get('access_code');
    $this->database = $database;
    $this->messenger = $messenger;
    $this->accessCodeManager = $manager;
    $this->userFloodControl = $user_flood_control;
  }

  /**
   * @inheritdoc
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('logger.factory'),
      $container->get('database'),
      $container->get('messenger'),
      $container->get('access_code.manager'),
      $container->get('user.flood_control')
    );
  }

  /**
   * Page callback for the use code link.
   */
  public function useCode($access_code, Request $request) {
    $flood_config = $this->config('user.flood');

    if (!$this->userFloodControl->isAllowed('user.failed_login_ip', $flood_config->get('ip_limit'), $flood_config->get('ip_window'))) {
      throw new AccessDeniedHttpException();
    }

    $uid = $this->accessCodeManager->validateAccessCode($access_code);

    if ($uid) {
      $user = User::load($uid);

      $url = $this->accessCodeManager->processLogin($user);
      return new RedirectResponse($url->toString());
    }
    else {
      $this->userFloodControl->register('user.failed_login_ip', $flood_config->get('ip_window'));

      throw new AccessDeniedHttpException();
    }
  }

}
