<?php

namespace Drupal\access_code\Form;

use Drupal\access_code\Service\AccessCodeManager;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\user\Entity\User;
use Drupal\user\UserFloodControl;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Class LoginForm.
 */
class LoginForm extends FormBase {

  /**
   * @var \Drupal\Core\Extension\ModuleHandler
   */
  private $moduleHandler;

  /**
   * @var \Drupal\access_code\Service\AccessCodeManager
   */
  private $accessCodeManager;

  /**
   * @var \Drupal\Core\Config\Config|\Drupal\Core\Config\ImmutableConfig
   */
  private $config;

  /**
   * @var \Drupal\user\UserFloodControl
   */
  protected $userFloodControl;

  /**
   * {@inheritdoc}
   */
  public function __construct(ModuleHandlerInterface $handler, AccessCodeManager $manager, ConfigFactoryInterface $config_factory, UserFloodControl $user_flood_control) {
    $this->moduleHandler = $handler;
    $this->accessCodeManager = $manager;
    $this->config = $config_factory->get('access_code.settings');
    $this->userFloodControl = $user_flood_control;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('module_handler'),
      $container->get('access_code.manager'),
      $container->get('config.factory'),
      $container->get('user.flood_control')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'access_code_login_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $form['access_code'] = [
      '#type' => $this->config->get('display_input') ? 'textfield' : 'password',
      '#title' => t('Access code'),
      '#description' => t('Enter your access code to log in.'),
      '#size' => 20,
      '#maxlength' => 20,
      '#required' => TRUE,
    ];

    $form['actions'] = [
      '#type' => 'actions'
    ];

    $form['actions']['submit'] = [
      '#type' => 'submit',
      '#value' => t('Log in'),
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    $flood_config = $this->config('user.flood');

    if (!$this->userFloodControl->isAllowed('user.failed_login_ip', $flood_config->get('ip_limit'), $flood_config->get('ip_window'))) {
      $form_state->setErrorByName('access_code', $this->t('Too many failed login attempts from your IP address. This IP address is temporarily blocked.'));
      return;
    }

    $uid = $this->accessCodeManager->validateAccessCode($form_state->getValue('access_code'));

    if (!$uid) {
      $this->userFloodControl->register('user.failed_login_ip', $flood_config->get('ip_window'));
      $form_state->setErrorByName('access_code', $this->t('Invalid access code.'));
    }
    else {
      $form_state->set('uid', $uid);
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    if (empty($uid = $form_state->get('uid'))) {
      return;
    }

    $user = User::load($uid);

    $url = $this->accessCodeManager->processLogin($user);
    $form_state->setRedirectUrl($url);
  }

}
