<?php

namespace Drupal\access_code\Form;

use Drupal\access_code\Service\AccessCodeManager;
use Drupal\Core\Config\TypedConfigManagerInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\user\Entity\Role;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Configure access code settings.
 */
class SettingsForm extends ConfigFormBase {

  /**
   * @var \Drupal\access_code\Service\AccessCodeManager
   */
  protected $accessCodeManager;

  /**
   * Constructor.
   */
  public function __construct(ConfigFactoryInterface $config_factory, TypedConfigManagerInterface $typed_config_manager, AccessCodeManager $manager) {
    parent::__construct($config_factory, $typed_config_manager);
    $this->accessCodeManager = $manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('config.factory'),
      $container->get('config.typed'),
      $container->get('access_code.manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'access_code_settings_form';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return [
      'access_code.settings',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $form = parent::buildForm($form, $form_state);
    $config = $this->config('access_code.settings');

    $roles = Role::loadMultiple();
    $role_options = array_map(function(Role $item) { return $item->label(); }, $roles);
    unset($role_options[AccountInterface::ANONYMOUS_ROLE]);
    unset($role_options[AccountInterface::AUTHENTICATED_ROLE]);

    $form['auto_code'] = [
      '#type' => 'fieldset',
      '#title' => 'Access code generation',
    ];

    $form['auto_code']['auto_code_length'] = [
      '#type' => 'select',
      '#options' => [
        4 => 4,
        5 => 5,
        6 => 6,
        7 => 7,
        8 => 8,
        9 => 9,
        10 => 10,
        11 => 11,
        12 => 12
      ],
      '#title' => $this->t('Length of auto generated code'),
      '#default_value' => $config->get('auto_code_length') ?: 8,
      '#description' => $this->t('Total length of access code EXCLUDING the prefix.'),
    ];

    $form['auto_code']['auto_code_prefix'] = [
      '#type' => 'textfield',
      '#size' => 20,
      '#title' => $this->t('Code prefix'),
      '#default_value' => $config->get('auto_code_prefix'),
      '#description' => $this->t('Optional string of characters to append to the beginning of access codes.'),
    ];

    $form['auto_code']['auto_code_format'] = [
      '#type' => 'radios',
      '#title' => $this->t('Format of code'),
      '#options' => [
        'alpha' => $this->t('Alphanumeric'),
        'numbers' => $this->t('Numbers'),
        'letters' => $this->t('Letters'),
      ],
      '#default_value' => $config->get('auto_code_format') ?: 'alpha',
    ];

    $form['expiration_default'] = [
      '#type' => 'select',
      '#title' => $this->t('Default expiration period'),
      '#options' => [
        'none' => $this->t('None'),
        '1 day' => $this->t('One Day'),
        '1 week' => $this->t('One Week'),
        '1 month' => $this->t('One Month'),
        '3 months' => $this->t('Three Months'),
        '6 months' => $this->t('Six Months'),
        '1 year' => $this->t('One Year'),
        '2 years' => $this->t('Two Years'),
        '3 years' => $this->t('Three Years'),
        '5 years' => $this->t('Five Years'),
        '10 years' => $this->t('Ten Years'),
      ],
      '#default_value' => $config->get('expiration_default') ?: '1 month',
      '#description' => $this->t('Automatically sets the expiration date a set period into the future when creating a new access code.'),
    ];

    $form['blocked_roles'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Blocked roles'),
      '#options' => $role_options,
      '#default_value' => $config->get('blocked_roles') ?: [],
      '#description' => $this->t('Roles that are not allowed to log in using access codes.'),
    ];

    $form['display_input'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Display input in access code field'),
      '#default_value' => $config->get('display_input'),
      '#description' => $this->t('Display the entered characters in the access code field when logging in.'),
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    parent::submitForm($form, $form_state);

    $config = $this->config('access_code.settings');
    $config
      ->set('auto_code_length', $form_state->getValue('auto_code_length'))
      ->set('auto_code_prefix', $form_state->getValue('auto_code_prefix'))
      ->set('auto_code_format', $form_state->getValue('auto_code_format'))
      ->set('expiration_default', $form_state->getValue('expiration_default'))
      ->set('blocked_roles', $form_state->getValue('blocked_roles'))
      ->set('display_input', $form_state->getValue('display_input'))
      ->save();
  }

}
