<?php

namespace Drupal\access_code\Service;

use Drupal\Component\Datetime\TimeInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Database\Connection;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Link;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\Routing\RedirectDestinationInterface;
use Drupal\Core\Session\AccountProxy;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\Url;
use Drupal\user\Entity\User;
use Drupal\user\UserDataInterface;
use Symfony\Component\HttpFoundation\RequestStack;

/**
 * Class AccessCodeManager.
 */
class AccessCodeManager {
  use StringTranslationTrait;

  /**
   * @var \Drupal\Core\Config\Config|\Drupal\Core\Config\ImmutableConfig
   */
  protected $config;

  /**
   * @var \Drupal\Core\Logger\LoggerChannelInterface
   */
  protected $logger;

  /**
   * @var \Drupal\Core\Database\Connection
   */
  protected $database;

  /**
   * @var \Drupal\Core\Session\AccountProxy
   */
  protected $currentUser;

  /**
   * @var \Drupal\user\UserDataInterface
   */
  protected $userData;

  /**
   * @var \Drupal\Component\Datetime\Time
   */
  protected $time;

  /**
   * @var \Drupal\Core\Extension\ModuleHandler
   */
  protected $moduleHandler;

  /**
   * @var \Symfony\Component\HttpFoundation\Request|null
   */
  protected $currentRequest;

  /**
   * @var \Drupal\Core\Routing\RedirectDestination
   */
  protected $redirectDestination;

  /**
   * @var \Drupal\Core\Messenger\Messenger
   */
  protected $messenger;

  /**
   * Constructor.
   */
  public function __construct(ConfigFactoryInterface $config_factory, LoggerChannelFactoryInterface $logger_factory, Connection $database, AccountProxy $current_user, UserDataInterface $user_data, TimeInterface $time, ModuleHandlerInterface $handler, RequestStack $stack, RedirectDestinationInterface $destination, MessengerInterface $messenger) {
    $this->config = $config_factory->get('access_code.settings');
    $this->logger = $logger_factory->get('access_code');
    $this->database = $database;
    $this->currentUser = $current_user;
    $this->userData = $user_data;
    $this->time = $time;
    $this->moduleHandler = $handler;
    $this->currentRequest = $stack->getCurrentRequest();
    $this->redirectDestination = $destination;
    $this->messenger = $messenger;
  }

  /**
   * Retrieves the access code for a given user.
   *
   * @param $uid
   *   User id.
   *
   * @return mixed|null
   *   The access code array containing 'code' and 'expiration', or NULL if not found.
   */
  public function getAccessCode($uid) {
    $result = $this->database->select('access_code', 'a')
      ->fields('a', ['code', 'expiration'])
      ->condition('uid', $uid)
      ->execute()
      ->fetchObject();

    if ($result) {
      return [
        'code' => $result->code,
        'expiration' => $result->expiration
      ];
    }

    return NULL;
  }

  /**
   * Validates an access code.
   *
   * @param $code
   *   Access code.
   *
   * @return
   *   The corresponding user id, if the code is valid; FALSE otherwise.
   */
  function validateAccessCode($code) {
    $access_code = $this->database->select('access_code')
      ->fields('access_code')
      ->condition('code', $code)
      ->execute()
      ->fetchAssoc();

    if (!$access_code) {
      return FALSE;
    }

    if (!empty($access_code['expiration']) && $access_code['expiration'] < $this->time->getCurrentTime()) {
      return FALSE;
    }

    /* @var \Drupal\user\Entity\User $user */
    $user = User::load($access_code['uid']);
    if (!$user) {
      return FALSE;
    }

    $blocked_roles = $this->config->get('blocked_roles');
    $is_blocked = $blocked_roles && array_intersect($user->getRoles(TRUE), $blocked_roles);

    return (!$is_blocked && $user->isActive()) ? $user->id() : FALSE;
  }

  /**
   * Adds access code fields to a form definition.
   */
  function addFormFields(&$form, FormStateInterface $form_state) {
    /* @var \Drupal\user\AccountForm $form_object */
    $form_object = $form_state->getFormObject();
    /* @var \Drupal\user\Entity\User $user */
    $user = $form_object->getEntity();

    $access_code = $user ? $this->getAccessCode($user->id()) : NULL;
    $blocked_roles = $this->config->get('blocked_roles');
    $is_blocked = $blocked_roles && array_intersect($user->getRoles(TRUE), $blocked_roles);

    $access = ($this->currentUser->id() == $user->id() && $this->currentUser->hasPermission('change own access code')) || $this->currentUser->hasPermission('administer users')  || $this->currentUser->hasPermission('change any access code');
    $can_pick_code = $this->currentUser->hasPermission('administer users');

    $form['account']['access_code_settings'] = [
      '#type' => 'fieldset',
      '#title' => t('Access code'),
      '#access' => $access,
      '#disabled' => $is_blocked,
      '#weight' => 100,
      '#prefix' => '<div id="access_code_settings-wrapper">',
      '#suffix' => '</div>',
    ];

    $form['account']['access_code_settings']['blocked_note'] = [
      '#markup' => '<i>' . $this->t('Access codes are disabled for this role') . '</i>',
      '#access' => $is_blocked,
    ];

    $form['account']['access_code_settings']['access_code'] = [
      '#type' => 'textfield',
      '#title' => $this->t('User\'s access code'),
      '#size' => 20,
      '#maxlength' => 20,
      '#description' => $this->t("The access code this user can use to authenticate by entering the code at 'user/ac', or by visiting the access link url ('ac/[access code]'). The code is not case sensitive. If none given, the user cannot use this method to log in."),
      '#default_value' => $access_code ? $access_code['code'] : NULL,
      '#access' => $access,
      '#disabled' => !$can_pick_code,
      '#weight' => 10,
    ];

    if ($form_state->getTriggeringElement() && $form_state->getTriggeringElement()['#name'] == 'generate_code') {
      $form['account']['access_code_settings']['access_code']['#value'] = $this->generateRandomCode();
    }

    $form['account']['access_code_settings']['generate_code'] = [
      '#type' => 'button',
      '#name' => 'generate_code',
      '#description' => t('Automatically generates a unique access code.'),
      '#value' => t('Generate random code'),
      '#access' => $access && !$can_pick_code,
      '#weight' => 11,
      '#limit_validation_errors' => [],
      '#ajax' => [
        'callback' => [static::class, 'generateCodeAjaxCallback'],
        'wrapper' => 'access_code_settings-wrapper',
        'method'  => 'replaceWith'
      ],
    ];

    $expires = NULL;
    if ($user->isNew() && $access) {
      $expiration_default = $this->config->get('expiration_default');

      if ($expiration_default && $expiration_default != 'none') {
        $date = new \DateTime('now');
        $date->modify('+' . $expiration_default);
        $expires = $date->format('Y-m-d');
      }
    }
    elseif (!empty($access_code['expiration'])) {
      $expires = date('Y-m-d', $access_code['expiration']);
    }

    $form['account']['access_code_settings']['access_code_expiration'] = [
      '#type' => 'date',
      '#title' => $this->t('Expiration'),
      '#description' => $this->t('Date after which the code cannot be used.'),
      '#default_value' => $expires,
      '#access' => $access,
      '#weight' => 20,
    ];

    if (!$user->isNew() && !empty($access_code)) {
      $login_url = Url::fromRoute('access_code.auto_login', ['access_code' => $access_code['code']], ['absolute' => TRUE]);
      $login_link = Link::fromTextAndUrl($login_url->toString(), $login_url);

      $form['account']['access_code_settings']['access_link'] = [
        '#markup' => t('Current access link:') . ' ' . $login_link->toString(),
        '#weight' => 30,
      ];
    }

    $form['#validate'][] = ['Drupal\access_code\Service\AccessCodeManager', 'validateFormFields'];
    $form['actions']['submit']['#submit'][] = ['Drupal\access_code\Service\AccessCodeManager', 'submitFormFields'];
  }

  /**
   * Submit handler to process the fields added to the user profile form.
   */
  public static function submitFormFields($form, FormStateInterface $form_state) {
    /* @var \Drupal\access_code\Service\AccessCodeManager $manager */
    $manager = \Drupal::service('access_code.manager');

    /* @var \Drupal\user\AccountForm $form_object */
    $form_object = $form_state->getFormObject();
    /* @var \Drupal\user\Entity\User $user */
    $user = $form_object->getEntity();

    $access_code = $form_state->getValue('access_code');
    $access_code_expiration = $form_state->getValue('access_code_expiration');

    // @TODO: Move this to hook_user_update().
    $manager->updateAccessCode($user, $access_code, $access_code_expiration);
  }

  /**
   * Ajax callback for the generate code button.
   */
  public static function generateCodeAjaxCallback($form, FormStateInterface $form_state) {
    return $form['account']['access_code_settings'];
  }

  /**
   * Validation handler for the user profile form.
   */
  public static function validateFormFields($form, FormStateInterface &$form_state) {
    /* @var \Drupal\access_code\Service\AccessCodeManager $manager */
    $manager = \Drupal::service('access_code.manager');

    /* @var \Drupal\user\AccountForm $form_object */
    $form_object = $form_state->getFormObject();
    /* @var \Drupal\user\Entity\User $user */
    $user = $form_object->getEntity();

    $invalid_chars = preg_replace('@[a-zA-Z0-9]@', '', $form_state->getValue('access_code'));
    if ($invalid_chars) {
      $form_state->setErrorByName('access_code', t('The access code can contain alphanumerical characters only.'));
    }

    if (strlen($form_state->getValue('access_code')) > 20) {
      $form_state->setErrorByName('access_code', t('The access code can not be longer than 20 characters.'));
    }

    if (!empty($form_state->getValue('access_code')) && strlen($form_state->getValue('access_code')) < 4) {
      $form_state->setErrorByName('access_code', t('Please choose an access code that is at least 4 characters long. A short access code poses security risk.'));
    }

    $access_code_expiration = $form_state->getValue('access_code_expiration');
    if ($access_code_expiration && strtotime($access_code_expiration) < \Drupal::service('datetime.time')->getCurrentTime()) {
      $form_state->setErrorByName('access_code_expiration', t('The expiration date must be in the future.'));
    }

    if (date('Y', strtotime($form_state->getValue('access_code_expiration'))) > 2037) {
      // Expiration is stored as a timestamp and dates beyond 2037 overflow when stored in the database.
      $form_state->setErrorByName('access_code_expiration', t('Expirations cannot be set beyond year 2037.'));
    }

    if (!empty($form_state->getValue('access_code'))) {
      $check = $manager->checkUniqueCode($form_state->getValue('access_code'), $user->id());

      if (!$check) {
        $form_state->setErrorByName('access_code', t('This access code has already been used. You cannot reuse access codes. Please enter a different code.'));
      }
    }
  }

  /**
   * Check if a given access code is unique.
   *
   * @param $code
   *   Access code.
   * @param  $uid
   *   User id of the user for whom the code will be generated. The user's
   *   own access code will be excluded form the verification.
   *
   * @return
   *   FALSE if the access code is already assigned to a user; TRUE otherwise.
   */
  function checkUniqueCode($code, $uid = NULL) {
    if (isset($uid)) {
      $result = $this->database->select('access_code', 'a')
        ->fields('a', ['code'])
        ->condition('code', $code)
        ->condition('uid', $uid, '<>')
        ->execute()
        ->fetchField();
    }
    else {
      $result = $this->database->select('access_code', 'a')
        ->fields('a', ['code'])
        ->condition('code', $code)
        ->execute()
        ->fetchField();
    }

    return $result ? FALSE : TRUE;
  }

  /**
   * Generates a random access code.
   *
   * @return
   *   The generated code.
   */
  function generateRandomCode($length = NULL, $format = NULL, $prefix = NULL) {
    $length = $length ?? ($this->config->get('auto_code_length') ?: 8);
    $format = $format ?? ($this->config->get('auto_code_format') ?: 'alpha');
    $prefix = $prefix ?? ($this->config->get('auto_code_prefix') ?: '');

    // Length must be minimum of 4 characters.
    if ($length < 4) {
      $length = 4;
    }

    switch ($format) {
      case 'numbers':
        $chars = '0123456789';
        break;
      case 'letters':
        $chars = 'ABCDEFGHIJKLMNOPQRSTUVWXYZ';
        break;
      case 'alpha':
      default:
        $chars = '0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ';
        break;
    }

    $passes = 0;
    $success = FALSE;

    do {
      $passes++;
      $random = '';

      for ($i = 0; $i < $length; $i++) {
        $random .= substr(str_shuffle($chars), 0, 1);
      }

      $access_code = $prefix . $random;

      $success = $this->checkUniqueCode($access_code);

    } while (!$success && $passes < 10);

    if (!$success) {
      $this->logger->error('Unique code could not be generated. Please check Access Code settings.');
      return NULL;
    }

    return $access_code;
  }

  /**
   * Logs in the user that has been successfully authenticated with an access
   * code.
   *
   * @param \Drupal\user\Entity\User $user
   *   The user.
   *
   * @return \Drupal\Core\Url
   *   The url that the user should be redirected to.
   */
  public function processLogin(User $user) {
    user_login_finalize($user);

    $this->messenger->addMessage($this->t('You have been logged in as %name.', ['%name' => $user->getAccountName()]));

    // Let modules determine the redirect destination.
    // Note: hook_access_code_login_redirect needs to return a Url object.
    $redirect_result = $this->moduleHandler->invokeAll('access_code_login_redirect', [$user]);

    if (!empty($redirect_result)) {
      /* @var \Drupal\Core\Url $url */
      $url = array_pop($redirect_result);
      return $url;
    }
    else if (!$this->currentRequest->query->has('destination')) {
      return Url::fromRoute('entity.user.canonical', ['user' => $user->id()]);
    }
    else {
      return Url::fromUserInput($this->redirectDestination->get());
    }
  }

  /**
   * Updates the access code for a user.
   *
   * @param \Drupal\user\Entity\User $user
   * @param $access_code
   * @param $expiration
   *
   * @return void
   */
  public function updateAccessCode(User $user, $access_code, $expiration = NULL) {
    if ($access_code) {
      $this->database->merge('access_code')
        ->keys(['uid' => $user->id()])
        ->fields([
          'code' => $access_code,
          'expiration' => $expiration ? strtotime($expiration) : 0,
        ])
        ->execute();
    }
    else {
      $this->database->delete('access_code')
        ->condition('uid', $user->id())
        ->execute();
    }
  }
}
