<?php

namespace Drupal\access_job_reporting\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Logger\LoggerChannelInterface;
use GuzzleHttp\ClientInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\JsonResponse;

/**
 * Provides an endpoint to fetch suggested ACCESS resource names.
 */
class ResourceListController extends ControllerBase {

  /**
   * HTTP client for outbound requests.
   */
  protected ClientInterface $httpClient;

  /**
   * Logger channel for this module.
   */
  protected LoggerChannelInterface $logger;

  /**
   * Constructs a ResourceListController.
   *
   * @param \GuzzleHttp\ClientInterface $httpClient
   *   HTTP client service.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $loggerFactory
   *   Logger channel for this module.
   */
  public function __construct(ClientInterface $httpClient, LoggerChannelFactoryInterface $loggerFactory) {
    $this->httpClient = $httpClient;
    $this->logger = $loggerFactory->get('access_job_reporting');
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): self {
    return new static(
      $container->get('http_client'),
      $container->get('logger.factory'),
    );
  }

  /**
   * Returns JSON containing suggested ACCESS resource names.
   */
  public function resourceList() {
    $config = $this->config('access_job_reporting.settings');
    $endpoint = (string) ($config->get('endpoint_url') ?? '');
    if ($endpoint === '') {
      $endpoint = 'https://allocations-api.access-ci.org/acdb/gateway/v2/job_attributes';
    }
    $apiKey = (string) ($config->get('api_key') ?? '');
    // Support Key module: use selected key value if configured.
    try {
      if (\Drupal::moduleHandler()->moduleExists('key')) {
        $keyName = (string) ($config->get('api_key_key') ?? '');
        if ($keyName !== '') {
          /** @var \Drupal\key\KeyRepositoryInterface $repo */
          $repo = \Drupal::service('key.repository');
          if ($repo) {
            $key = $repo->getKey($keyName);
            if ($key) {
              $value = $key->getKeyValue();
              if (is_string($value) && $value !== '') {
                $apiKey = $value;
              }
            }
          }
        }
      }
    }
    catch (\Throwable $e) {
      // Ignore and use config-stored apiKey.
    }
    $agent = (string) ($config->get('agent_name') ?? '');

    $resources = [];
    try {
      $headers = [];
      if ($apiKey !== '') {
        $headers['XA-API-KEY'] = $apiKey;
      }
      if ($agent !== '') {
        $headers['XA-AGENT'] = $agent;
      }

      // Trigger the suggestion list by posting an unknown resource name.
      $response = $this->httpClient->request('POST', $endpoint, [
        'headers' => $headers,
        'form_params' => [
          'xsederesourcename' => 'unknown',
        ],
        'http_errors' => FALSE,
        'timeout' => 10,
      ]);

      $body = (string) $response->getBody();
      $resources = $this->parseSuggestedResources($body);
    }
    catch (\Throwable $e) {
      $this->logger->warning('Failed to fetch ACCESS resources: @msg', ['@msg' => $e->getMessage()]);
    }

    if (empty($resources)) {
      // Fallback minimal list.
      $resources = [
        'expanse.sdsc.xsede.org',
        'bridges2.psc.xsede.org',
      ];
    }

    return new JsonResponse([
      'resources' => $resources,
      'endpoint' => $endpoint,
      'source' => 'ACCESS',
    ]);
  }

  /**
   * Parse suggested resource names from ACCESS response text.
   *
   * @param string $text
   *   Raw response text from the ACCESS job_attributes API.
   *
   * @return array
   *   An array of resource name strings.
   */
  private function parseSuggestedResources(string $text): array {
    $lines = preg_split('/\r?\n/', $text);
    $collect = FALSE;
    $results = [];
    foreach ($lines as $line) {
      $trim = trim($line);
      if (str_contains($trim, 'Did you mean one of these?')) {
        $collect = TRUE;
        continue;
      }
      if ($collect) {
        if ($trim === '' || str_starts_with($trim, '*')) {
          break;
        }
        // Lines are prefixed with two spaces in examples; just trim.
        if ($trim !== '') {
          $results[] = $trim;
        }
      }
    }
    return $results;
  }

}
