<?php

namespace Drupal\access_job_reporting\Form;

use Drupal\access_job_reporting\AccessResourceFetcher;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Configuration form for ACCESS Job Reporting.
 */
class AccessJobReportingForm extends ConfigFormBase {

  /**
   * Resource fetcher service.
   */
  protected AccessResourceFetcher $resourceFetcher;

  /**
   * Constructs the form.
   */
  public function __construct($config_factory, AccessResourceFetcher $resource_fetcher) {
    parent::__construct($config_factory);
    $this->resourceFetcher = $resource_fetcher;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('config.factory'),
      $container->get('access_job_reporting.resource_fetcher'),
    );
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['access_job_reporting.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'access_job_reporting_settings_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('access_job_reporting.settings');

    // Intro text at the top of the form.
    $form['intro'] = [
      '#type' => 'markup',
      '#prefix' => '<div class="access-job-reporting-intro" style="margin-bottom: 1rem;">',
      '#suffix' => '</div>',
      '#markup' => $this->t(
        'Reports TAPIS job metadata to the <a href=":url" target="_blank" rel="noopener noreferrer">ACCESS-CI Allocations API</a>, standardizing job reporting and tracking science gateway usage of ACCESS resources.',
        [':url' => 'https://allocations-api.access-ci.org/acdb/']
      ),
    ];

    $form['endpoint_url'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Endpoint URL (job_attributes)'),
      '#default_value' => $config->get('endpoint_url') ?? 'https://allocations-api.access-ci.org/acdb/gateway/v2/job_attributes',
      '#description' => $this->t('Use the ACCESS job reporting endpoint: @url', [
        '@url' => 'https://allocations-api.access-ci.org/acdb/gateway/v2/job_attributes',
      ]),
      '#placeholder' => 'https://allocations-api.access-ci.org/acdb/gateway/v2/job_attributes',
    ];

    // If Key module is available and keys exist, prefer selecting a Key.
    $has_key_module = \Drupal::moduleHandler()->moduleExists('key');
    $key_options = [];
    if ($has_key_module) {
      try {
        $keys = \Drupal::entityTypeManager()->getStorage('key')->loadMultiple();
        foreach ($keys as $key_entity) {
          $key_options[$key_entity->id()] = $key_entity->label();
        }
      }
      catch (\Throwable $e) {
        $key_options = [];
      }
    }

    if ($has_key_module && !empty($key_options)) {
      $form['api_key_key'] = [
        '#type' => 'select',
        '#title' => $this->t('ACCESS API key'),
        '#options' => $key_options,
        '#default_value' => $config->get('api_key_key') ?? '',
        '#empty_option' => $this->t('- Select a key -'),
        '#required' => TRUE,
        '#description' => $this->t('Uses a Key from the Key module. The secret value is never stored in this module\'s config.'),
      ];
    }
    else {
      // Fallback to plain text input when Key module is absent or has no keys.
      $form['api_key'] = [
        '#type' => 'textfield',
        '#title' => $this->t('ACCESS API key'),
        '#default_value' => $config->get('api_key') ?? '',
        '#required' => TRUE,
      ];
    }

    $form['agent_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Agent name (XA-AGENT)'),
      '#default_value' => $config->get('agent_name') ?? '',
      '#required' => TRUE,
    ];

    $form['debug_mode'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable debug mode (no POSTs)'),
      '#description' => $this->t('When enabled, the worker does not send HTTP POSTs. It logs the payload for a dry run.'),
      '#default_value' => (bool) ($config->get('debug_mode') ?? FALSE),
    ];

    $form['retry_interval'] = [
      '#type' => 'number',
      '#title' => $this->t('Retry interval (seconds)'),
      '#default_value' => (int) ($config->get('retry_interval') ?? 86400),
      '#min' => 0,
      '#required' => TRUE,
    ];

    $form['max_attempts'] = [
      '#type' => 'number',
      '#title' => $this->t('Max attempts'),
      '#default_value' => (int) ($config->get('max_attempts') ?? 15),
      '#min' => 1,
      '#required' => TRUE,
    ];

    $form['resources'] = [
      '#type' => 'details',
      '#title' => $this->t('ACCESS resources'),
      '#open' => FALSE,
    ];

    // Wrapper for AJAX updates.
    $form['resources']['list'] = [
      '#type' => 'container',
      '#prefix' => '<div id="access-job-reporting-resources-wrapper">',
      '#suffix' => '</div>',
    ];

    // Inline reference list fetched from controller JSON.
    $resources = $this->getResourcesFromFetcher();
    $form['resources']['list']['resource_list'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Suggested resource names'),
      '#default_value' => !empty($resources) ? implode("\n", $resources) : '',
      '#rows' => 12,
      '#disabled' => TRUE,
      '#description' => $this->t(''),
      '#description_display' => 'after',
    ];

    // AJAX button to refresh the list inline.
    $form['resources']['fetch_ajax'] = [
      '#type' => 'button',
      '#value' => $this->t('Reload ACCESS Resources'),
      '#ajax' => [
        'callback' => '::ajaxFetchResources',
        'event' => 'click',
        'wrapper' => 'access-job-reporting-resources-wrapper',
      ],
      '#limit_validation_errors' => [],
      '#attributes' => ['class' => ['button', 'button--primary']],
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $editable = $this->configFactory->getEditable('access_job_reporting.settings');

    // Support both Key module selection and direct entry fallback.
    if ($form_state->hasValue('api_key_key')) {
      $selected_key = (string) $form_state->getValue('api_key_key');
      $editable->set('api_key_key', $selected_key);
      // Clear raw api_key to avoid storing secrets when using Key module.
      if ($selected_key !== '') {
        $editable->set('api_key', '');
      }
    }
    else {
      $editable->set('api_key', (string) $form_state->getValue('api_key'));
      // Ensure we don’t mistakenly keep an old key selection.
      $editable->set('api_key_key', '');
    }

    $editable
      ->set('agent_name', $form_state->getValue('agent_name'))
      ->set('debug_mode', (bool) $form_state->getValue('debug_mode'))
      ->set('retry_interval', (int) $form_state->getValue('retry_interval'))
      ->set('max_attempts', (int) $form_state->getValue('max_attempts'))
      ->set('endpoint_url', (string) $form_state->getValue('endpoint_url'))
      ->save();

    parent::submitForm($form, $form_state);
  }

  /**
   * AJAX callback to refresh the resource list inline.
   */
  public function ajaxFetchResources(array &$form, FormStateInterface $form_state) {
    $form_state->setRebuild(TRUE);
    $resources = $this->getResourcesFromFetcher();

    $form['resources']['#open'] = TRUE;
    $form['resources']['list']['resource_list']['#value'] = !empty($resources) ? implode("\n", $resources) : '';

    return $form['resources']['list'];
  }

  /**
   * Helper to fetch resources from the external API.
   */
  private function getResourcesFromFetcher(): array {
    $result = $this->resourceFetcher->fetch();
    return is_array($result['resources'] ?? NULL) ? $result['resources'] : [];
  }

}
