<?php

namespace Drupal\access_job_reporting;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Queue\QueueFactory;
use GuzzleHttp\ClientInterface;

/**
 * Core service to enqueue TAPIS job reports to ACCESS.
 */
class AccessJobReporter {

  const SETTINGS = 'access_job_reporting.settings';
  const QUEUE_NAME = 'access_job_reporting.job_queue';

  /**
   * Config factory service.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected ConfigFactoryInterface $configFactory;

  /**
   * Queue factory service.
   *
   * @var \Drupal\Core\Queue\QueueFactory
   */
  protected QueueFactory $queueFactory;

  /**
   * HTTP client service.
   *
   * @var \GuzzleHttp\ClientInterface
   */
  protected ClientInterface $httpClient;

  /**
   * Logger channel for this module.
   *
   * @var \Drupal\Core\Logger\LoggerChannelInterface
   */
  protected $logger;

  /**
   * Constructs the reporter service.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   Config factory.
   * @param \Drupal\Core\Queue\QueueFactory $queueFactory
   *   Queue factory.
   * @param \GuzzleHttp\ClientInterface $httpClient
   *   HTTP client.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $loggerFactory
   *   Logger factory.
   */
  public function __construct(
    ConfigFactoryInterface $configFactory,
    QueueFactory $queueFactory,
    ClientInterface $httpClient,
    LoggerChannelFactoryInterface $loggerFactory
  ) {
    $this->configFactory = $configFactory;
    $this->queueFactory = $queueFactory;
    $this->httpClient = $httpClient;
    $this->logger = $loggerFactory->get('access_job_reporting');
  }

  /**
   * Enqueue a job to be reported.
   */
  public function enqueue(
    string $jobId,
    string $jobUuid,
    string $tenantId,
    string $jobOwnerId,
    string $resource,
    string $queueName,
    string $allocation,
    string $accountUuid,
    string $gatewayUser,
    string $submitTime,
    string $software,
    string $remoteJobId): void {
    $queue = $this->queueFactory->get(self::QUEUE_NAME);
    $queue->createItem([
      'tapis_job_id' => $jobId,
      'tapis_job_uuid' => $jobUuid,
      'tenant_id' => $tenantId,
      'job_owner_id' => $jobOwnerId,
      'remote_job_id' => $remoteJobId,
      'account_uuid' => $accountUuid,
      'gateway_user' => $gatewayUser,
      'submittime' => $submitTime,
      'software' => $software,
      'allocation' => $allocation,
      'resource' => $resource,
      'queue' => $queueName,
      'attempt_count' => 0,
      'last_attempt' => 0,
      'status' => 'queued',
    ]);
  }

  /**
   * Parse resource map text into [queueName => resourceName].
   */
  public function buildQueueToResourceMap(string $mapText): array {
    $mapping = [];
    $seen_sources = [];
    foreach (preg_split('/\r?\n/', $mapText) as $line) {
      $line = trim($line);
      if ($line === '' || str_starts_with($line, '#')) {
        continue;
      }
      $parts = array_map('trim', explode('|', $line, 2));
      if (count($parts) !== 2) {
        continue;
      }
      [$resource, $queuesCsv] = $parts;
      foreach (explode(',', $queuesCsv) as $q) {
        $q = trim($q);
        if ($q === '') {
          continue;
        }
        // Normalize queue keys to lowercase for case-insensitive matching.
        $key = strtolower($q);
        if (isset($mapping[$key]) && $mapping[$key] !== $resource) {
          // Log duplicate queue mapping; keep the first mapping.
          $this->logger->warning(
            'Duplicate queue mapping detected for "@queue". Keeping resource "@keep" and ignoring "@ignore".',
            [
              '@queue' => $q,
              '@keep' => $mapping[$key],
              '@ignore' => $resource,
            ]
          );
          $seen_sources[$key][] = $resource;
          continue;
        }
        $mapping[$key] = $resource;
        $seen_sources[$key][] = $resource;
      }
    }
    return $mapping;
  }

}
