<?php

namespace Drupal\access_job_reporting;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Logger\LoggerChannelInterface;
use GuzzleHttp\ClientInterface;

/**
 * Fetches suggested ACCESS resource names.
 */
class AccessResourceFetcher {

  /**
   * Default fallback resources when the API returns nothing.
   */
  private const FALLBACK_RESOURCES = [
    'expanse.sdsc.xsede.org',
    'bridges2.psc.xsede.org',
  ];

  protected ConfigFactoryInterface $configFactory;

  protected ClientInterface $httpClient;

  protected LoggerChannelInterface $logger;

  protected ModuleHandlerInterface $moduleHandler;

  /**
   * Optional key repository service.
   *
   * @var object|null
   */
  protected $keyRepository;

  /**
   * Constructs the fetcher.
   */
  public function __construct(
    ConfigFactoryInterface $configFactory,
    ClientInterface $httpClient,
    LoggerChannelFactoryInterface $loggerFactory,
    ModuleHandlerInterface $moduleHandler,
    $keyRepository = NULL
  ) {
    $this->configFactory = $configFactory;
    $this->httpClient = $httpClient;
    $this->logger = $loggerFactory->get('access_job_reporting');
    $this->moduleHandler = $moduleHandler;
    $this->keyRepository = $keyRepository;
  }

  /**
   * Fetch resources from ACCESS API.
   *
   * @return array<string, mixed>
   *   Keys: resources (array) and endpoint (string).
   */
  public function fetch(): array {
    $config = $this->configFactory->get('access_job_reporting.settings');
    $endpoint = (string) ($config->get('endpoint_url') ?? '');
    if ($endpoint === '') {
      $endpoint = 'https://allocations-api.access-ci.org/acdb/gateway/v2/job_attributes';
    }
    $apiKey = (string) ($config->get('api_key') ?? '');

    // Prefer secret managed by Key module when available.
    if ($this->moduleHandler->moduleExists('key') && $this->keyRepository) {
      try {
        $keyName = (string) ($config->get('api_key_key') ?? '');
        if ($keyName !== '') {
          $key = $this->keyRepository->getKey($keyName);
          if ($key) {
            $value = $key->getKeyValue();
            if (is_string($value) && $value !== '') {
              $apiKey = $value;
            }
          }
        }
      }
      catch (\Throwable $e) {
        // Fall back to config-stored API key if Key module fails.
      }
    }
    $agent = (string) ($config->get('agent_name') ?? '');

    $resources = [];

    try {
      $headers = [];
      if ($apiKey !== '') {
        $headers['XA-API-KEY'] = $apiKey;
      }
      if ($agent !== '') {
        $headers['XA-AGENT'] = $agent;
      }

      $response = $this->httpClient->request('POST', $endpoint, [
        'headers' => $headers,
        'form_params' => [
          'xsederesourcename' => 'unknown',
        ],
        'http_errors' => FALSE,
        'timeout' => 10,
      ]);

      $body = (string) $response->getBody();
      $resources = $this->parseSuggestedResources($body);

      if (empty($resources)) {
        $this->logger->notice('ACCESS suggestion list returned no resources.');
      }
    }
    catch (\Throwable $e) {
      $this->logger->warning('Failed to fetch ACCESS resources: @msg', ['@msg' => $e->getMessage()]);
    }

    if (empty($resources)) {
      $resources = self::FALLBACK_RESOURCES;
    }

    return [
      'resources' => $resources,
      'endpoint' => $endpoint,
    ];
  }

  /**
   * Parse suggested resources from ACCESS API response text.
   *
   * @param string $text
   *   Response body from ACCESS.
   *
   * @return array<int, string>
   *   List of suggested resource names.
   */
  private function parseSuggestedResources(string $text): array {
    $lines = preg_split('/\r?\n/', $text) ?: [];
    $collect = FALSE;
    $results = [];
    foreach ($lines as $line) {
      $trim = trim($line);
      if (str_contains($trim, 'Did you mean one of these?')) {
        $collect = TRUE;
        continue;
      }
      if ($collect) {
        if ($trim === '' || str_starts_with($trim, '*')) {
          break;
        }
        if ($trim !== '') {
          $results[] = $trim;
        }
      }
    }
    return $results;
  }

}
