<?php

namespace Drupal\access_job_reporting\Plugin\QueueWorker;

use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Queue\QueueWorkerBase;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\tapis_job\TapisProvider\TapisJobProviderInterface;
use GuzzleHttp\ClientInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Queue worker that reports job metadata to ACCESS.
 *
 * @QueueWorker(
 *   id = "access_job_reporting.job_queue",
 *   title = @Translation("ACCESS Job Reporting queue worker"),
 *   cron = {"time" = 20}
 * )
 */
class AccessJobQueueWorker extends QueueWorkerBase implements ContainerFactoryPluginInterface {

  const PROD_ENDPOINT = 'https://allocations-api.access-ci.org/acdb/gateway/v2/job_attributes';

  /**
   * HTTP client for outbound ACCESS calls.
   *
   * @var \GuzzleHttp\ClientInterface
   */
  protected ClientInterface $httpClient;

  /**
   * Logger channel for this module.
   *
   * @var \Drupal\Core\Logger\LoggerChannelInterface
   */
  protected $logger;

  /**
   * Config factory service.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * TAPIS Job provider.
   *
   * @var \Drupal\tapis_job\TapisProvider\TapisJobProviderInterface
   */
  protected TapisJobProviderInterface $tapisJobProvider;

  /**
   * Constructs the queue worker.
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    ClientInterface $httpClient,
    LoggerChannelFactoryInterface $loggerFactory,
    ConfigFactoryInterface $configFactory,
    TapisJobProviderInterface $tapisJobProvider
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->httpClient = $httpClient;
    $this->logger = $loggerFactory->get('access_job_reporting');
    $this->configFactory = $configFactory;
    $this->tapisJobProvider = $tapisJobProvider;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('http_client'),
      $container->get('logger.factory'),
      $container->get('config.factory'),
      $container->get('tapis_job.tapis_job_provider'),
    );
  }

  /**
   * Determine if we have enough info to look up the TAPIS job.
   */
  protected function canLookupTapisJob(array $data): bool {
    return !empty($data['tenant_id'])
      && !empty($data['job_owner_id'])
      && !empty($data['tapis_job_uuid']);
  }

  /**
   * Populate missing fields from TAPIS job metadata.
   *
   * @return bool
   *   TRUE if a lookup was attempted.
   */
  protected function refreshFromTapisJob(array $data, string &$remoteJobId, string &$submitTime): bool {
    try {
      $tapisJob = $this->tapisJobProvider->getJob(
        (string) $data['tenant_id'],
        (string) $data['tapis_job_uuid'],
        (string) $data['job_owner_id']
      );
      if ($remoteJobId === '' && !empty($tapisJob['remoteJobId'])) {
        $remoteJobId = (string) $tapisJob['remoteJobId'];
      }
      if ($submitTime === '') {
        $submitRaw = $tapisJob['remoteSubmitted'] ?? $tapisJob['created'] ?? '';
        if ($submitRaw !== '') {
          $ts = is_numeric($submitRaw) ? (int) $submitRaw : (strtotime((string) $submitRaw) ?: 0);
          if ($ts > 0) {
            $submitTime = date('Y-m-d H:i T', $ts);
          }
        }
      }
      return TRUE;
    }
    catch (\Throwable $e) {
      $this->logger->notice('Unable to refresh TAPIS job @uuid for ACCESS reporting: @msg', [
        '@uuid' => (string) ($data['tapis_job_uuid'] ?? ''),
        '@msg' => $e->getMessage(),
      ]);
      return TRUE;
    }
  }

  /**
   * Processes a queue item by posting job attributes to ACCESS.
   */
  public function processItem($data) {
    $this->logger->info('data: @data', ['@data' => json_encode($data, JSON_PRETTY_PRINT)]);
    $config = $this->configFactory->get('access_job_reporting.settings');
    $apiKey = (string) $config->get('api_key');
    // If Key module is present and a key is selected, use its secret value.
    try {
      if (\Drupal::moduleHandler()->moduleExists('key')) {
        $keyName = (string) ($config->get('api_key_key') ?? '');
        if ($keyName !== '') {
          /** @var \Drupal\key\KeyRepositoryInterface $repo */
          $repo = \Drupal::service('key.repository');
          if ($repo) {
            $key = $repo->getKey($keyName);
            if ($key) {
              $value = $key->getKeyValue();
              if (is_string($value) && $value !== '') {
                $apiKey = $value;
              }
            }
          }
        }
      }
    }
    catch (\Throwable $e) {
      // Fall back to config-stored api_key if key module integration fails.
    }
    $agentName = (string) $config->get('agent_name');
    $retryInterval = (int) $config->get('retry_interval');
    if ($retryInterval <= 0) {
      $retryInterval = 86400; // Default: 1 day.
    }
    $maxAttempts = (int) $config->get('max_attempts');
    if ($maxAttempts <= 0) {
      $maxAttempts = 15; // Default: 15 attempts.
    }
    $debug = (bool) $config->get('debug_mode');

    $attempt = (int) ($data['attempt_count'] ?? 0);
    $last = (int) ($data['last_attempt'] ?? 0);
    if ($attempt > 0 && (time() - $last) < $retryInterval) {
      // Requeue without processing until retry window expires.
      throw new \Exception('Waiting before retry per retry_interval.');
    }

    $headers = [
      'XA-API-KEY' => $apiKey,
    ];
    if ($agentName !== '') {
      $headers['XA-AGENT'] = $agentName;
    }

    // If missing, try a lightweight fetch.
    $remoteJobId = (string) ($data['remote_job_id'] ?? '');
    $submitTime = (string) ($data['submittime'] ?? '');
    $lookupAttempted = FALSE;
    if ($remoteJobId === '' && $this->canLookupTapisJob($data)) {
      $lookupAttempted = $this->refreshFromTapisJob($data, $remoteJobId, $submitTime);
    }
    // If only submit time is missing, try once as well.
    if ($submitTime === '' && !$lookupAttempted && $this->canLookupTapisJob($data)) {
      $lookupAttempted = $this->refreshFromTapisJob($data, $remoteJobId, $submitTime);
    }
    // If still no remote id, skip this item after a refresh attempt.
    if ($remoteJobId === '') {
      $this->logger->warning(
        $lookupAttempted
          ? 'Skipping ACCESS report for TAPIS job @uuid because remote job id is missing after TAPIS refresh; removing queue item.'
          : 'Skipping ACCESS report for TAPIS job @uuid because remote job id is missing and TAPIS refresh could not be attempted; removing queue item.',
        ['@uuid' => (string) ($data['tapis_job_uuid'] ?? '')]
      );
      return;
    }

    $resourceName = (string) ($data['resource'] ?? '');
    $this->logger->info('RESOURCE NAME: @resourceName', ['@resourceName' => $resourceName]);
    $formParams = [
      'gatewayuser' => (string) ($data['gateway_user'] ?? ''),
      'xsederesourcename' => $resourceName,
      'jobid' => $remoteJobId,
      'submittime' => $submitTime,
    ];
    if (!empty($data['software'])) {
      $formParams['software'] = (string) $data['software'];
    }

    if ($debug) {
      $this->logger->notice(
          'Debug mode: ACCESS job for TAPIS job uuid: @tapis_job_uuid, is not reported.',
          ['@tapis_job_uuid' => $data['tapis_job_uuid']]
          );
      $formParams['debug'] = 'x';
    }

    try {
      $endpoint = (string) ($config->get('endpoint_url') ?? '');
      if ($endpoint === '') {
        $endpoint = self::PROD_ENDPOINT;
      }
      // 'headers' => $headers,
      $response = $this->httpClient->request('POST', $endpoint, [
        'headers' => $headers,
        'form_params' => $formParams,
        'http_errors' => FALSE,
      ]);
      $status = $response->getStatusCode();
      $this->logger->info('ACCESS job report status: @status', ['@status' => $status]);
      if ($status >= 200 && $status < 300) {
        $body = (string) $response->getBody();
        $this->logger->info('ACCESS job report body: @body', ['@body' => $body]);
        $this->logger->info('Reported job @job to ACCESS', ['@job' => $formParams['jobid']]);
        return;
        // Throw new \Exception('ACCESS job report failed. Please try again later.');.
      }
      // Failure; increment attempts and rethrow to requeue.
      $body = (string) $response->getBody();
      $this->logger->info('ACCESS job report body: @body', ['@body' => $body]);
      $attempt++;
      if ($attempt >= $maxAttempts) {
        $this->logger->error(
          'ACCESS reporting failed permanently for job @job: @code @body',
          [
            '@job' => $formParams['jobid'],
            '@code' => $status,
            '@body' => $body,
          ]
        );
        return;
      }
      $data['attempt_count'] = $attempt;
      $data['last_attempt'] = time();
      // Throwing exception causes the item to be requeued by cron runner.
      throw new \Exception('ACCESS reporting failed: ' . $status);
    }
    catch (\Throwable $e) {
      // Update attempt info and requeue by throwing.
      $attempt++;
      if ($attempt >= $maxAttempts) {
        $this->logger->error(
          'ACCESS reporting failed permanently for job @job: @msg',
          [
            '@job' => (string) ($data['remote_job_id'] ?? $data['tapis_job_id'] ?? ''),
            '@msg' => $e->getMessage(),
          ]
        );
        return;
      }
      $data['attempt_count'] = $attempt;
      $data['last_attempt'] = time();
      throw $e;
    }
  }

}
