<?php

namespace Drupal\access_unpublished;

use Drupal\Component\Utility\Html;
use Drupal\Core\Datetime\DateFormatterInterface;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityListBuilder;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Url;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Defines a class to build a listing of access token entities.
 */
class AccessTokenListBuilder extends EntityListBuilder {

  /**
   * The date formatter service.
   *
   * @var \Drupal\Core\Datetime\DateFormatterInterface
   */
  protected $dateFormatter;

  /**
   * The Access Token Manager.
   *
   * @var \Drupal\access_unpublished\AccessTokenManager
   */
  protected $accessTokenManager;

  /**
   * Name of the list builder.
   *
   * @var string
   */
  protected $handlerName = 'list_builder';

  /**
   * {@inheritdoc}
   */
  public static function createInstance(ContainerInterface $container, EntityTypeInterface $entity_type) {
    $builder = parent::createInstance($container, $entity_type);
    $builder->setDateFormatter($container->get('date.formatter'));
    $builder->setAccessTokenManager($container->get('access_unpublished.access_token_manager'));
    return $builder;
  }

  /**
   * Sets the date formatter service.
   *
   * @param \Drupal\Core\Datetime\DateFormatterInterface $dateFormatter
   *   The date formatter service.
   */
  protected function setDateFormatter(DateFormatterInterface $dateFormatter) {
    $this->dateFormatter = $dateFormatter;
  }

  /**
   * Sets the access token manager.
   *
   * @param \Drupal\access_unpublished\AccessTokenManager $accessTokenManager
   *   The access manager.
   */
  protected function setAccessTokenManager(AccessTokenManager $accessTokenManager) {
    $this->accessTokenManager = $accessTokenManager;
  }

  /**
   * {@inheritdoc}
   */
  public function buildHeader() {
    // Enable language column and filter if multiple languages are added.
    $header = [
      'expire_date' => $this->t('Expire date'),
      'host' => $this->t('Parent entity'),
      'label' => $this->t('Label'),
    ];
    return $header + parent::buildHeader();
  }

  /**
   * {@inheritdoc}
   */
  public function buildRow(EntityInterface $entity) {
    /** @var \Drupal\access_unpublished\Entity\AccessToken $entity */

    $row['expire_date']['data'] = [
      '#plain_text' => $entity->get('expire')->value > 0 ? $this->dateFormatter->format($entity->get('expire')->value, 'short') : $this->t('Unlimited'),
    ];
    if ($entity->isExpired()) {
      $row['expire_date']['data'] = [
        '#markup' => 'Expired token',
        '#prefix' => '<div class="access-unpublished-expired">',
        '#suffix' => '</div>',
      ];
    }

    if ($hostEntity = $entity->getHost()) {
      $row['host']['data'] = [
        '#type' => 'link',
        '#title' => $hostEntity->label(),
        '#url' => $hostEntity->toUrl(),
      ];
    }

    $row['label']['data'] = [
      '#plain_text' => $entity->get('label')->value ? $entity->get('label')->value : '',
    ];

    $row['operations']['data'] = $this->buildOperations($entity);

    return $row + parent::buildRow($entity);
  }

  /**
   * {@inheritdoc}
   */
  public function buildOperations(EntityInterface $entity) {
    $build = [
      '#type' => 'operations',
      '#links' => $this->getOperations($entity),
      '#attached' => [
        'library' => ['access_unpublished/drupal.access_unpublished.admin'],
      ],
    ];

    // Make the link AJAX on list controller pages
    // only for renew and delete operations.
    foreach ($build["#links"] as $key => $link) {
      if ($key === 'renew' || $key === 'delete') {
        if (isset($build["#links"][$key]['url'])) {
          $url = &$build["#links"][$key]['url'];
          $url_query = $url->getOption('query') ?: [];
          $url_query['handler'] = $this->handlerName;
          $url = $url->setOption('query', $url_query);
        }

        $build["#links"][$key]['attributes']['class'][] = 'use-ajax';
      }
    }

    return $build;
  }

  /**
   * {@inheritdoc}
   */
  protected function getDefaultOperations(EntityInterface $token) {
    /** @var \Drupal\access_unpublished\AccessTokenInterface $token */

    $operations = parent::getDefaultOperations($token);
    if ($token->access('delete') && $token->hasLinkTemplate('delete')) {
      $operations['delete'] = [
        'title' => $this->t('Delete'),
        'url' => $this->ensureDestination($token->toUrl('delete')),
        'weight' => 100,
      ];
    }
    if ($token->access('renew') && $token->isExpired()) {
      $operations['renew'] = [
        'title' => $this->t('Renew'),
        'url' => $this->ensureDestination($token->toUrl('renew')),
        'weight' => 50,
      ];
    }
    elseif ($hostEntity = $token->getHost()) {
      $url = $this->accessTokenManager->getAccessTokenUrl($token, $hostEntity->language());
      $operations['copy'] = [
        'title' => $this->t('Copy'),
        'url' => Url::fromUserInput('#'),
        'attributes' => [
          'data-unpublished-access-url' => $url,
          'class' => ['clipboard-button'],
        ],
        'weight' => 50,
      ];
    }
    return $operations;
  }

  /**
   * {@inheritdoc}
   */
  public function render() {
    $build = parent::render();
    $build['table']['#id'] = Html::getUniqueId('access-token-list');
    $build['table']['#attributes']['data-drupal-selector'] = Html::getId('access-token-list');
    return $build;
  }

}
