<?php

namespace Drupal\access_unpublished\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;

/**
 * Configure access unpublished settings for this site.
 */
class SettingsForm extends ConfigFormBase {

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return [
      'access_unpublished.settings',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'settings_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('access_unpublished.settings');

    $form['hash_key'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Hash key'),
      '#default_value' => $config->get('hash_key'),
      '#size' => 60,
      '#maxlength' => 128,
      '#required' => TRUE,
    ];

    $form['duration'] = [
      '#type' => 'select',
      '#title' => $this->t('Lifetime'),
      '#description' => $this->t('Default lifetime of the generated access tokens.'),
      '#options' => AccessUnpublishedForm::getDurationOptions(),
      '#default_value' => $config->get('duration'),
    ];

    $form['label'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Default token label'),
      '#description' => $this->t('The default label to use when generating access tokens.'),
      '#default_value' => $config->get('label'),
      '#size' => 60,
      '#maxlength' => 128,
    ];

    $form['cleanup_tokens_on_publish'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Cleanup tokens on publish'),
      '#description' => $this->t('Tokens are cleared when content is published'),
      '#default_value' => $config->get('cleanup_tokens_on_publish'),
    ];

    $form['cleanup_expired_tokens'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Cleanup expired tokens'),
      '#description' => $this->t('Cron will cleanup expired tokens.'),
      '#default_value' => $config->get('cleanup_expired_tokens'),
    ];

    $form['cleanup_expired_tokens_period'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Period of time for cron cleanup.'),
      '#default_value' => $config->get('cleanup_expired_tokens_period'),
      '#description' => $this->t("Describe a time by reference to the current day, like '-90 days' (All tokens which expired more than 90 days ago). See <a href=\"http://php.net/manual/function.strtotime.php\">strtotime</a> for more details."),
      '#size' => 60,
      '#maxlength' => 128,
      '#states' => [
        'visible' => [
          ':input[name="cleanup_expired_tokens"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $form['modify_http_headers'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Modify HTTP Headers on unpublished page.'),
      '#default_value' => $this->prepareHeadersDisplay(),
      '#description' => $this->t("Enter one header per line, in the format key|label."),
      '#element_validate' => [[static::class, 'validateHttpHeaders']],
      '#field_has_data' => FALSE,
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    parent::submitForm($form, $form_state);

    $this->config('access_unpublished.settings')
      ->set('hash_key', $form_state->getValue('hash_key'))
      ->set('label', $form_state->getValue('label'))
      ->set('duration', $form_state->getValue('duration'))
      ->set('cleanup_tokens_on_publish', $form_state->getValue('cleanup_tokens_on_publish'))
      ->set('cleanup_expired_tokens', $form_state->getValue('cleanup_expired_tokens'))
      ->set('cleanup_expired_tokens_period', $form_state->getValue('cleanup_expired_tokens_period'))
      ->set('modify_http_headers', $form_state->getValue('modify_http_headers'))
      ->save();
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    $strtotime = @strtotime($form_state->getValue('cleanup_expired_tokens_period'));
    if (!$strtotime) {
      $form_state->setErrorByName('cleanup_expired_tokens_period', $this->t('The relative start date value entered is invalid.'));
    }
    elseif ($strtotime > time()) {
      $form_state->setErrorByName('cleanup_expired_tokens_period', $this->t('The value has to be negative.'));
    }
  }

  /**
   * Element validation callback for the modify_http_headers field.
   *
   * @param array $element
   *   An associative array containing the properties and children of the
   *   generic form element.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current state of the form for the form this element belongs to.
   *
   * @see \Drupal\Core\Render\Element\FormElement::processPattern()
   */
  public static function validateHttpHeaders($element, FormStateInterface $form_state) {
    $values = [];

    $value = $element['#value'];

    $lines = explode("\n", $value);
    $errors = [];

    foreach ($lines as $line_number => $line) {
      $line = trim($line);

      // Skip empty lines.
      if ($line === '') {
        continue;
      }

      // Check if line contains the pipe separator.
      if (strpos($line, '|') === FALSE) {
        $errors[] = t('Line @line: Missing pipe separator (|). Format should be "Header-Name|value".', [
          '@line' => $line_number + 1,
        ]);
        continue;
      }

      // Split by pipe.
      $parts = explode('|', $line, 2);
      $header_name = trim($parts[0]);
      $header_value = isset($parts[1]) ? trim($parts[1]) : '';

      // Validate header name (RFC 7230)
      if (empty($header_name)) {
        $errors[] = t('Line @line: Header name cannot be empty.', [
          '@line' => $line_number + 1,
        ]);
      }
      elseif (!preg_match('/^[a-zA-Z0-9!#$%&\'*+.^_`|~-]+$/', $header_name)) {
        $errors[] = t('Line @line: Invalid header name "@name". Header names can only contain letters, numbers, and certain special characters.', [
          '@line' => $line_number + 1,
          '@name' => $header_name,
        ]);
      }

      // Validate header value (check for control characters except tab)
      if (preg_match('/[\x00-\x08\x0A-\x1F\x7F]/', $header_value)) {
        $errors[] = t('Line @line: Header value contains invalid control characters.', [
          '@line' => $line_number + 1,
        ]);
      }

      $values[$header_name] = $header_value;
    }

    // Set form error if any validation errors found.
    if (!empty($errors)) {
      $form_state->setError($element, [
        '#theme' => 'item_list',
        '#items' => $errors,
        '#title' => t('HTTP Headers validation errors:'),
      ]);
    }

    $form_state->setValueForElement($element, $values);
  }

  /**
   * Format array to display on settings form.
   *
   * @return string
   *   Return string of HTTP headers.
   */
  private function prepareHeadersDisplay() {
    $headers = $this->config('access_unpublished.settings')
      ->get('modify_http_headers');

    $lines = [];
    if (!empty($headers) && is_array($headers)) {
      foreach ($headers as $key => $value) {
        $lines[] = "$key|$value";
      }
    }
    return implode("\n", $lines);
  }

}
