<?php

namespace Drupal\Tests\access_unpublished\Functional;

use Drupal\access_unpublished\Entity\AccessToken;
use Drupal\node\Entity\Node;
use Drupal\node\Entity\NodeType;
use Drupal\Tests\BrowserTestBase;
use Drupal\Tests\node\Traits\NodeCreationTrait;
use Drupal\user\RoleInterface;
use Drupal\views\Tests\ViewTestData;

/**
 * Tests the relationship between AccessToken and content entities in Views.
 *
 * @group access_unpublished
 */
class AccessTokenRelationshipTest extends BrowserTestBase {

  use NodeCreationTrait;

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * Modules to enable.
   *
   * @var array
   */
  protected static $modules = [
    'node',
    'access_unpublished',
    'views_ui',
    'access_unpublished_test_views',
  ];

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Install the test views.
    ViewTestData::createTestViews(get_class($this), ['access_unpublished_test_views']);

    // Create a 'page' content type.
    NodeType::create(['type' => 'page', 'name' => 'Page'])->save();

    // Grant 'access content' permission to anonymous users.
    user_role_grant_permissions(RoleInterface::ANONYMOUS_ID, [
      'access content',
    ]);

    // Log in a user with 'administer views' permission.
    $this->drupalLogin($this->drupalCreateUser([
      'administer views',
    ]));
  }

  /**
   * Tests the relationship between AccessToken and Node entities in Views.
   */
  public function testAccessTokenNodeRelationship() {
    $assert_session = $this->assertSession();

    // Create a test node.
    $node = Node::create([
      'type' => 'page',
      'title' => 'Test Node Title for Relationship',
      'status' => 1,
    ]);
    $node->save();

    // Create an access token for the test node.
    $accessToken = AccessToken::create([
      'entity_type' => $node->getEntityTypeId(),
      'entity_id' => $node->id(),
      'value' => 'test_token_value_functional',
      'expire' => \Drupal::time()->getRequestTime() + 3600,
    ]);
    $accessToken->save();

    // Navigate to the view's page.
    $this->drupalGet('test-access-token-relationship');
    $assert_session->statusCodeEquals(200);

    // Assert that the node's title and the access token's value are displayed.
    $assert_session->pageTextContains($node->getTitle());
    $assert_session->pageTextContains($accessToken->get('value')->value);
  }

}
