<?php

namespace Drupal\Tests\access_unpublished\Functional;

use Drupal\access_unpublished\Entity\AccessToken;
use Drupal\node\Entity\NodeType;
use Drupal\Tests\BrowserTestBase;
use Drupal\Tests\node\Traits\NodeCreationTrait;
use Drupal\user\RoleInterface;
use Drupal\views\Tests\ViewTestData;

/**
 * Tests the views integration.
 *
 * @group access_unpublished
 */
class AccessUnpublishedViewsTest extends BrowserTestBase {

  use NodeCreationTrait;

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * Modules to enable.
   *
   * @var array
   */
  protected static $modules = [
    'node',
    'access_unpublished',
    'views_ui',
    'access_unpublished_test_views',
  ];

  /**
   * {@inheritdoc}
   */
  protected function setUp($import_test_views = TRUE): void {
    parent::setUp();

    ViewTestData::createTestViews(get_class($this), ['access_unpublished_test_views']);

    NodeType::create(['type' => 'page', 'name' => 'page'])->save();

    user_role_grant_permissions(RoleInterface::ANONYMOUS_ID, [
      'access content',
    ]);

    $this->drupalLogin($this->drupalCreateUser([
      'administer views',
      'delete token',
    ]));
  }

  /**
   * Tests the fields.
   *
   * @throws \Drupal\Core\Entity\EntityStorageException
   * @throws \Behat\Mink\Exception\ExpectationException
   */
  public function testFields(): void {
    $assert_session = $this->assertSession();

    $node = $this->createNode(['status' => FALSE]);
    $token = AccessToken::create([
      'entity_type' => 'node',
      'entity_id' => $node->id(),
    ]);
    $token->save();

    $this->drupalGet('test-access-unpublished-tokens');
    $assert_session->statusCodeEquals(200);

    // Check that the view title is displayed on page.
    $assert_session->pageTextContains('test_access_unpublished_tokens');

    // Generate the expected access token copy URL for the node.
    $expected_copy_url = $node->toUrl(
      'canonical',
      [
        'query' => ['auHash' => $token->get('value')->value],
        'absolute' => TRUE,
      ],
    )->toString();

    // Check that the node URL with the access token is displayed on page.
    $assert_session->pageTextContains($expected_copy_url);

    // Check for the "Copy" link and its data attribute.
    $copy_link = $assert_session->elementExists('css', 'a.clipboard-button:contains("Copy")');
    $this->assertEquals($expected_copy_url, $copy_link->getAttribute('data-unpublished-access-url'));

    // Generate the expected access token delete URL.
    $expected_token_delete_url = $token->toUrl('delete')->toString();

    // The user has the "delete token" permission, check for the delete link.
    $assert_session->linkByHrefExists($expected_token_delete_url);
  }

}
