<?php

namespace Drupal\Tests\access_unpublished\Functional;

use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\node\Entity\NodeType;
use Drupal\Tests\BrowserTestBase;
use Drupal\Tests\node\Traits\NodeCreationTrait;
use Drupal\Tests\workspaces\Functional\WorkspaceTestUtilities;
use Drupal\user\RoleInterface;

/**
 * Tests the token generator UI in a Workspaces context.
 *
 * @group access_unpublished
 */
class WorkspacesAccessTest extends BrowserTestBase {

  use NodeCreationTrait;
  use WorkspaceTestUtilities;
  use StringTranslationTrait;

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'access_unpublished',
    'node',
    'workspaces',
  ];

  /**
   * The entity to test.
   *
   * @var \Drupal\node\Entity\Node
   */
  protected $entity;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    NodeType::create(['type' => 'page', 'name' => 'page'])->save();

    user_role_grant_permissions(RoleInterface::ANONYMOUS_ID, [
      'access content',
      'access_unpublished node page',
    ]);

    // Create an unpublished entity.
    $this->entity = $this->createNode(['status' => FALSE]);

    $assert_session = $this->assertSession();

    // Verify that the entity is not accessible.
    $this->drupalGet($this->entity->toUrl());
    $assert_session->statusCodeEquals(403);
  }

  /**
   * The access_unpublished form element should appear when not in a workspace.
   */
  public function testAccessUnpublishedFormElementPresent() {
    $assert_session = $this->assertSession();

    // Create a user with permission to edit page nodes.
    $editor = $this->drupalCreateUser([
      'access content',
      'edit any page content',
      'access_unpublished node page',
    ]);

    // Log in as the editor.
    $this->drupalLogin($editor);

    // Navigate to the node edit form.
    $this->drupalGet($this->entity->toUrl('edit-form'));

    // Verify the page loaded successfully.
    $assert_session->statusCodeEquals(200);

    // Verify the access_unpublished form element is present.
    $assert_session->elementExists('css', '.access-unpublished-form input[type=submit][value="Generate token"]');
  }

  /**
   * The access_unpublished form element should not appear when in a workspace.
   */
  public function testAccessUnpublishedFormElementHiddenInWorkspace() {
    $assert_session = $this->assertSession();

    // Create a user with permissions to edit pages and manage workspaces.
    $editor = $this->drupalCreateUser([
      'access content',
      'edit any page content',
      'access_unpublished node page',
      'administer workspaces',
      'create workspace',
      'edit own workspace',
      'view own workspace',
    ]);

    // Log in as the editor.
    $this->drupalLogin($editor);

    // Create and activate a workspace.
    $workspace_label = 'Test Workspace';
    $this->createAndActivateWorkspaceThroughUi($workspace_label, 'test_workspace');

    // Navigate to the node edit form while in the workspace.
    $this->drupalGet($this->entity->toUrl('edit-form'));

    // Verify the page loaded successfully.
    $assert_session->statusCodeEquals(200);

    // Verify the access_unpublished "Generate token" button is NOT present.
    $assert_session->elementNotExists('css', '.access-unpublished-form input[type=submit][value="Generate token"]');
    // Instead, we should see a message that indicates why it's not there.
    $expected_message = 'You are currently in workspace ' . $workspace_label . '. The temporary unpublished access module does not work well inside workspaces.';
    $assert_session->pageTextContains($expected_message);
  }

}
