<?php

namespace Drupal\access_unpublished_group\Plugin;

use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\group\Plugin\GroupContentAccessControlHandler;

/**
 * Custom access controller for group entities to support Access Unpublished.
 */
class AccessUnpublishedGroupContentAccessControlHandler extends GroupContentAccessControlHandler {

  /**
   * {@inheritdoc}
   */
  public function entityAccess(EntityInterface $entity, $operation, AccountInterface $account, $return_as_object = FALSE) {
    $result = parent::entityAccess($entity, $operation, $account, TRUE);
    if ($operation == 'view' && $result->isForbidden()) {
      /** @var \Drupal\group\Entity\Storage\GroupContentStorageInterface $storage */
      $storage = $this->entityTypeManager->getStorage('group_content');
      $group_contents = $storage->loadByEntity($entity);

      // Find content that uses this plugin and has the group permission.
      $has_group_permission = FALSE;
      foreach ($group_contents as $group_content) {
        $plugin_id = $group_content->getContentPlugin()->getPluginId();
        if ($plugin_id === $this->pluginId) {
          $group = $group_content->getGroup();
          $permission_id = 'access_unpublished_group_' . $group_content->getContentPlugin()->getPluginId();
          if ($group->hasPermission($permission_id, $account)) {
            $has_group_permission = TRUE;
            break;
          }
        }
      }

      if ($has_group_permission) {
        // Check the access_unpublished_entity_access() once when the user has
        // the group permission.
        $au_access = access_unpublished_entity_access($entity, $operation, $account);
        if ($au_access->isAllowed()) {
          // Add the cache dependencies from the parent result.
          $au_access->addCacheableDependency($result);
          $result = $au_access;
        }
        else {
          // Add the access_unpublished_entity_access cache dependencies to the
          // returned result.
          $result->addCacheableDependency($au_access);
        }
      }
    }
    return $return_as_object ? $result : $result->isAllowed();
  }

}
