<?php

namespace Drupal\access_unpublished_group\Access;

use Drupal\group\Plugin\Group\Relation\GroupRelationInterface;
use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\group\Plugin\Group\Relation\GroupRelationTypeManagerInterface;

/**
 * Provides dynamic permissions for groups of different types.
 */
class AccessUnpublishedGroupPermissions implements ContainerInjectionInterface {

  /**
   * The route match.
   *
   * @var \Drupal\Core\Routing\RouteMatchInterface
   */
  protected $routeMatch;

  /**
   * The group content enabler plugin manager.
   *
   * @var \Drupal\group\Plugin\Group\Relation\GroupRelationTypeManagerInterface
   */
  protected $groupRelationTypeManager;

  /**
   * Constructs the AccessUnpublishedGroupPermissions object.
   */
  public function __construct(RouteMatchInterface $route_match, GroupRelationTypeManagerInterface $group_relation_type_manager) {
    $this->routeMatch = $route_match;
    $this->groupRelationTypeManager = $group_relation_type_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('current_route_match'),
      $container->get('group_relation_type.manager')
    );
  }

  /**
   * Returns an array of group permissions.
   *
   * @return array
   *   The group permissions.
   *   @see \Drupal\user\PermissionHandlerInterface::getPermissions()
   */
  public function groupPermissions() {
    $perms = [];
    if (is_null($this->groupRelationTypeManager)) {
      return [];
    }
    if ($group_type = $this->routeMatch->getParameter('group_type')) {
      $plugins = $this->groupRelationTypeManager->getInstalled($group_type);
      foreach ($plugins as $plugin) {
        $perms += $this->buildPermissions($plugin);
      }
    }
    return $perms;
  }

  /**
   * Returns list of group permissions for a given group content enabler plugin.
   *
   * @param \Drupal\group\Plugin\Group\Relation\GroupRelationInterface $plugin
   *   The group content enabler plugin.
   *
   * @return array
   *   An associative array of permission names and descriptions.
   */
  protected function buildPermissions(GroupRelationInterface $plugin) {
    $permissions = [];
    $permissions['access_unpublished_group_' . $plugin->getPluginId()] = [
      'title' => 'Access unpublished ' . $plugin->getPluginDefinition()->get('label') ?? '',
    ];
    return $permissions;
  }

}
