<?php

namespace Drupal\access_unpublished_group\Plugin\Group\RelationHandler;

use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\group\Plugin\Group\RelationHandler\AccessControlInterface;
use Drupal\group\Plugin\Group\RelationHandler\AccessControlTrait;

/**
 * Provides an access control handler to support Access Unpublished.
 */
class AccessUnpublishedGroupAccessControl implements AccessControlInterface {

  use AccessControlTrait;

  /**
   * Constructs a new AccessUnpublishedGroupAccessControl.
   *
   * @param \Drupal\group\Plugin\Group\RelationHandler\AccessControlInterface $parent
   *   The parent access control handler.
   */
  public function __construct(AccessControlInterface $parent) {
    $this->parent = $parent;
  }

  /**
   * {@inheritdoc}
   */
  public function entityAccess(EntityInterface $entity, $operation, AccountInterface $account, $return_as_object = FALSE) {
    $result = $this->parent->entityAccess($entity, $operation, $account, $return_as_object);

    if ($operation == 'view' && $result->isForbidden()) {
      /** @var \Drupal\group\Entity\Storage\GroupRelationshipStorageInterface $storage */
      $storage = $this->entityTypeManager->getStorage('group_content');
      $group_relationships = $storage->loadByEntity($entity);

      // Find content that uses this plugin and has the group permission.
      $has_group_permission = FALSE;
      foreach ($group_relationships as $group_relationship) {
        $plugin_id = $group_relationship->getPlugin()->getRelationTypeId();
        if ($plugin_id === $this->pluginId) {
          $group = $group_relationship->getGroup();
          $permission_id = 'access_unpublished_group_' . $plugin_id;
          if ($group->hasPermission($permission_id, $account)) {
            $has_group_permission = TRUE;
            break;
          }
        }
      }

      if ($has_group_permission) {
        // Check the access_unpublished_entity_access() once when the user has
        // the group permission.
        $au_access = access_unpublished_entity_access($entity, $operation, $account);
        if ($au_access->isAllowed()) {
          // Add the cache dependencies from the parent result.
          $au_access->addCacheableDependency($result);
          $result = $au_access;
        }
        else {
          // Add the access_unpublished_entity_access cache dependencies to the
          // returned result.
          $result->addCacheableDependency($au_access);
        }
      }
    }

    return $return_as_object ? $result : $result->isAllowed();
  }

}
